"""Class representing a generic model."""

from __future__ import annotations

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
import ast
import copy
from datetime import datetime
from enum import auto, Enum, Flag, unique
from sys import float_info
from typing import Any, Iterable, Optional, Sequence, TypeAlias

# 2. Third party modules
from packaging.version import parse as parse_version

# 3. Aquaveo modules
from xms.guipy.data.target_type import TargetType
from xms.guipy.widgets.table_with_tool_bar import TableDefinition

# 4. Local modules
from xms.gmi.data.pretty_printer import pretty_print

MIN_FLOAT = -float_info.max
MAX_FLOAT = float_info.max


@unique
class Type(Enum):
    """Type names for parameters in a GenericModel."""
    BOOLEAN = auto()
    CHECKBOX_CURVE = auto()
    CURVE = auto()
    DATASET = auto()
    DATE_TIME = auto()
    FLOAT = auto()
    FLOAT_CURVE = auto()  # Internal format is (mode, float, xs, ys)
    FLOAT_OR_CURVE = auto()
    FONT = auto()
    INPUT_FILE = auto()
    INPUT_FOLDER = auto()
    INTEGER = auto()
    OPTION = auto()
    OUTPUT_FILE = auto()
    SERIES = auto()
    TEXT = auto()
    TABLE = auto()

    def __str__(self):
        """
        Convert the `Type` to a string suitable for serialization.

        The returned string can be used as an index into `Type` to recover the
        original `Type`. In other words, if `t` is an instance of `Type`, then
        `Type[str(t)] == t`.
        """
        return self.name


class Curve(Flag):
    """
    Modes for curve parameters.

    These can be combined with the | operator to enable multiple options at
    once. Combining `CONSTANT` with others won't normally make sense.
    """
    _CURVE = auto()
    _DATES = auto()
    _STAIRS = auto()

    CONSTANT = auto()  #: The curve has the same value everywhere. It is constant.
    CURVE = _CURVE  #: The curve's value varies. It is a curve.
    DATES = _CURVE | _DATES  #: Use dates on the x-axis.
    STAIRS = _CURVE | _STAIRS  #: Use stair-step interpolation instead of linear.

    def __str__(self):
        """
        Convert the `Curve` to a string suitable for serialization.

        The string can be passed to `Curve.from_string()` to deserialize the `Curve`.
        """
        name_map = {
            Curve.CONSTANT: 'CONSTANT',
            Curve.CURVE: 'CURVE',
            Curve.DATES: 'DATES',
            Curve.STAIRS: 'STAIRS',
        }
        names = [name_map[item] for item in name_map.keys() if item in self]
        return '|'.join(names)

    @classmethod
    def from_string(cls, value: str) -> Curve:
        """
        Parse a string into a `Curve`.

        The inverse of `str(self)`.

        Args:
            value: A string returned by `str(self)`.

        Returns:
            An instance of `Curve` equivalent to the one used to produce the string.
        """
        pieces = value.split('|')
        value = Curve[pieces.pop()]
        while pieces:
            value |= Curve[pieces.pop()]
        return value


UNASSIGNED_MATERIAL_ID = '0'
UNASSIGNED_MATERIAL_NAME = 'Unassigned'

# The .2dm format names things using integers, but most other code prefers naming things with strings.
GmiName: TypeAlias = str | int

_PRE_VERSION_NUMBERS = 0
_ADDED_VERSION_NUMBERS = 1
_RENAMED_CURVE_TO_SERIES_AND_ADDED_NEW_CURVE_TYPE = 2
_REMOVED_VERSION_AUTO_UPDATE = 3
_MADE_FLOAT_CURVE_FIXED_SIZE_VALUE = 4
_LATEST_VERSION = _MADE_FLOAT_CURVE_FIXED_SIZE_VALUE


class Parameter:
    """
    A class representing a parameter.

    Parameter includes all the information about a parameter, including its
    current value.

    `Parameter` instances should not be created directly. See the various
    `Group.add_*()` methods for alternatives meant for external use.

    Note that there may be multiple "copies" of a parameter. For example, each
    point or arc may have its own value for a parameter. An instance of this
    class can only hold one value at a time, so if multiple materials or
    boundaries are needed, then multiple instances will be needed too (or one
    will have to be saved off to put another in here -- see
    `Section.extract_values`).
    """
    def __init__(self, section_name: GmiName, group_name: GmiName, parameter_name: GmiName, parameter):
        """
        Initialize the class.

        This is mainly for internal use by `Group`. See the various
        `Group.add_*()` methods for alternatives meant for external use.

        Args:
            section_name: Name of the section the parameter is in.
            group_name: Name of the group the parameter is in.
            parameter_name: Name of the parameter.
            parameter: A dictionary representing the parameter.
        """
        self.section_name = section_name
        self.group_name = group_name
        self.parameter_name = parameter_name
        self._parameter = parameter

    @staticmethod
    def create(parameter_type: Type, label: str, default, description: str = '', required: bool = False) -> dict:
        """
        Create a new Parameter dictionary suitable for passing to `self.__init__()`.

        This is mainly for internal use by `Group`. See the various
        `Group.add_*()` methods for alternatives meant for external use.

        Args:
            parameter_type: Type of the parameter. See `Type`.
            label: User-friendly label for the parameter.
            default: Default value.
            description: A short piece of text providing extra detail about
                the parameter. Typically one or two sentences.
            required: Whether the parameter requires the user to set a value.

        Returns:
            A Parameter dictionary suitable for passing to self.__init__().
        """
        value = {'type': parameter_type, 'label': label, 'default': default, 'required': required}
        if description:
            value['description'] = description
        return value

    @property
    def label(self) -> str:
        """
        The parameter's label.

        The label is a short name suitable for display to the user.
        """
        return self._parameter['label']

    @property
    def checkbox_label(self) -> str:
        """
        The parameter's checkbox's label.

        The label is a short name suitable for display to the user.
        """
        return self._parameter['checkbox_label']

    @checkbox_label.setter
    def checkbox_label(self, value: str):
        """
        The parameter's checkbox's label.

        The label is a short name suitable for display to the user.
        """
        self._parameter['checkbox_label'] = value

    @property
    def axis_titles(self) -> list[str]:
        """The axis titles for a curve or float-curve parameter."""
        return self._parameter['axis_titles']

    @axis_titles.setter
    def axis_titles(self, value: list[str]):
        """The axis titles for a curve or float-curve parameter."""
        self._parameter['axis_titles'] = value

    @property
    def default(self):
        """The parameter's default value."""
        return self._parameter['default']

    @default.setter
    def default(self, value):
        """The parameter's default value."""
        self._parameter['default'] = value

    @property
    def description(self) -> str:
        """The parameter's description."""
        return self._parameter.get('description', '')

    @property
    def checkbox_description(self) -> str:
        """Description for a checkbox-curve's checkbox's description."""
        return self._parameter.get('checkbox_description', '')

    @checkbox_description.setter
    def checkbox_description(self, value: str):
        self._parameter['checkbox_description'] = value

    @property
    def options(self) -> list[str]:
        """The allowed options for option types."""
        return self._parameter['options']

    @options.setter
    def options(self, value: list[str]):
        """The allowed options for option types."""
        self._parameter['options'] = value

    @property
    def file_filter(self) -> str:
        """The file filter for file parameters."""
        return self._parameter['file_filter']

    @file_filter.setter
    def file_filter(self, value: str):
        """Set the file filter for file parameters."""
        self._parameter['file_filter'] = value

    @property
    def high(self) -> int | float:
        """The upper bound for numeric types."""
        return self._parameter.get('high', None)

    @high.setter
    def high(self, value: int | float):
        """Set the upper bound for numeric types."""
        self._parameter['high'] = value

    @property
    def low(self) -> int | float:
        """The lower bound for numeric types."""
        return self._parameter.get('low', None)

    @low.setter
    def low(self, value: int | float):
        """Set the lower bound for numeric types."""
        self._parameter['low'] = value

    @property
    def parameter_type(self) -> Type:
        """
        The parameter's type.

        A string from the Type class.
        """
        return self._parameter['type']

    @property
    def use_dates(self) -> bool:
        """Whether to use dates in the curve editor."""
        return self._parameter['use_dates']

    @use_dates.setter
    def use_dates(self, value: bool):
        """Whether to use dates in the curve editor."""
        self._parameter['use_dates'] = value

    @property
    def required(self) -> bool:
        """Whether a value for the parameter is required."""
        return self._parameter['required']

    @required.setter
    def required(self, value: bool):
        """Whether a value for the parameter is required."""
        self._parameter['required'] = value

    @property
    def table_definition(self):
        """Table column types."""
        return TableDefinition.from_dict(self._parameter['table_definition'])

    @table_definition.setter
    def table_definition(self, table_definition: TableDefinition):
        """Set table column types."""
        self._parameter['table_definition'] = table_definition.to_dict()

    @property
    def display_table(self):
        """Whether the table is displayed, or if a button is displayed which opens a table dialog."""
        return self._parameter['display_table']

    @display_table.setter
    def display_table(self, value: bool):
        """Whether the table is displayed, or if a button is displayed which opens a table dialog."""
        self._parameter['display_table'] = value

    @property
    def calendar_popup(self) -> bool:
        """Get whether the date/time parameter uses the calendar_popup option."""
        return self._parameter['calendar_popup']

    @calendar_popup.setter
    def calendar_popup(self, calendar_popup: bool):
        """Sets whether the date/time parameter uses the calendar_popup option."""
        self._parameter['calendar_popup'] = calendar_popup

    @property
    def modes(self) -> tuple[Curve, Curve]:
        """A value for how a checkbox-curve should behave when unchecked, and checked."""
        return self._parameter['modes']

    @modes.setter
    def modes(self, value: tuple[Curve, Curve]):
        """A value for how a checkbox-curve should behave when unchecked, and checked."""
        self._parameter['modes'] = value

    @property
    def mode(self) -> Curve:
        """Get the current mode."""
        mode_index = 1 if self.value[0] else 0
        return self.modes[mode_index]

    @mode.setter
    def mode(self, new_mode: bool):
        """Set the current mode with the new state of the checkbox."""
        mode_index = 1 if self.value[0] else 0
        old_mode = self.modes[mode_index]

        mode_index = 1 if new_mode else 0
        new_mode = self.modes[mode_index]

        if Curve.CURVE in old_mode and Curve.CONSTANT in new_mode:
            # Curve -> Constant; get the default constant
            self.value = (mode_index, self.default[1])
        elif Curve.CONSTANT in old_mode and Curve.CURVE in new_mode:
            # Constant -> Curve; use the default curve
            self.value = (mode_index, -1)
        else:
            # Not changing; keep the old value
            self.value = (mode_index, self.value[1])

    @property
    def value(self):
        """The parameter's current value."""
        if 'value' in self._parameter and self._parameter['value'] is not None:
            return self._parameter['value']
        return self._parameter['default']

    @value.setter
    def value(self, new_value):
        """Set the parameter's current value."""
        self._parameter['value'] = new_value

    @property
    def has_value(self) -> bool:
        """
        Whether the parameter has a value.

        `self.value` will return `self.default` if the parameter does not have
        a value. This can be used to tell whether it returned a real value or
        the default one.
        """
        return self._parameter.get('value', None) is not None

    def clear_value(self):
        """Clears the value."""
        self._parameter.pop('value', None)

    def add_dependency(self, parent: Parameter | list[str], flags: dict):
        """
        Make this parameter's availability depend on another parameter's value.

        Args:
            parent: The parameter this one depends on. This can be an instance
                of a `Parameter`, or a list of strings describing the "path" to
                the parent. The list option is maintained for compatibility
                with HGS, RSM, and SCHISM, which started using it before the
                `Parameter` option was added. The `Parameter` option is
                recommended since it is easy to get the parameter given the
                information needed to call this, and it is less tempting to
                hard-code the section name, which is an implementation detail.

                Lists should be `[section_name, group_name, parameter_name]`.
                `section_name` should be the string returned by `section.name`,
                where `section` is the `Section` containing the parameter.
                Then the parent will be looked up as
                `section.group(group_name).parameter(parameter_name)`.
            flags: Dictionary mapping (possible value of parent -> whether
                this parameter is enabled for that value). `Parameter` doesn't
                require that the dictionary's keys exactly match the parent's
                possible values, but consuming code might.
        """
        if isinstance(parent, list):
            self._parameter['parent'] = parent
        else:
            self._parameter['parent'] = [parent.section_name, parent.group_name, parent.parameter_name]
        self._parameter['dependency_flags'] = flags

    @property
    def parent(self):
        """The parameter's parent."""
        if 'parent' in self._parameter:
            return self._parameter['parent']
        return None

    @property
    def dependency_flags(self):
        """The parameter's dependency flags."""
        return self._parameter.get('dependency_flags', None)

    def compatible(self, other: Parameter) -> bool:
        """
        Check for compatibility between this Parameter and `other`.

        Two parameters are considered compatible if everything but their value
        is the same.

        Args:
            other: Parameter to compare against.

        Returns:
            Whether they are compatible.
        """
        difference = self._parameter.items() ^ other._parameter.items()
        for key, _ in difference:
            if key != 'value':
                return False
        return True

    def extract_value(self):
        """
        Retrieve the parameter's current value for serialization purposes.

        This is meant for internal use. External code should typically extract
        values at the `Section` level.

        Returns:
            The current value, if present, or `None`.
        """
        return self._parameter.get('value', self.default)

    def restore_value(self, value):
        """
        Restore the parameter's value.

        This is meant for internal use. External code should typically restore
        values at the `Section` level.
        """
        self._parameter['value'] = value

    def to_literal(self):
        """
        Convert the `Parameter` from an in-memory representation to a Python literal.

        This is typically called right before serialization.

        :meta private:  This is only public so `Group` can call it.
        """
        if self._parameter['type'] == Type.CHECKBOX_CURVE:
            self._parameter['modes'] = [str(mode) for mode in self._parameter['modes']]
        self._parameter['type'] = self._parameter['type'].name

    def from_literal(self):
        """
        Convert the `Parameter` from a literal to a more convenient in-memory representation.

        This is typically called right after deserialization.

        :meta private:  This is only public so `Group` can call it.
        """
        self._parameter['type'] = Type[self._parameter['type']]
        if self._parameter['type'] == Type.CHECKBOX_CURVE:
            self._parameter['modes'] = tuple(Curve.from_string(mode) for mode in self._parameter['modes'])

    def migrate(self, from_version):
        """
        Migrate the `Parameter` to the latest version.

        External code shouldn't have to call this.

        Args:
            from_version: The version being migrated from.
        """
        # Skip _PRE_VERSION_NUMBERS and _ADDED_VERSION_NUMBERS since nothing changed.

        if from_version < _RENAMED_CURVE_TO_SERIES_AND_ADDED_NEW_CURVE_TYPE:
            if self._parameter['type'] == 'CURVE':
                # This was renamed to match Group.add_xy_series() and make it available for Group.add_curve().
                self._parameter['type'] = 'SERIES'
            from_version = _RENAMED_CURVE_TO_SERIES_AND_ADDED_NEW_CURVE_TYPE

        # Skip _REMOVED_VERSION_AUTO_UPDATE since nothing changed.
        if from_version < _REMOVED_VERSION_AUTO_UPDATE:
            from_version = _REMOVED_VERSION_AUTO_UPDATE

        if from_version < _MADE_FLOAT_CURVE_FIXED_SIZE_VALUE:
            from_version = _MADE_FLOAT_CURVE_FIXED_SIZE_VALUE
            if self._parameter['type'] == 'FLOAT_CURVE':
                mode, float_value = self._parameter['default']
                self._parameter['default'] = (mode, float_value, [0.0], [0.0])
            if self._parameter['type'] == 'FLOAT_CURVE' and 'value' in self._parameter:
                mode = self._parameter['value'][0]
                if self._parameter['value'][0] == 'FLOAT':
                    float_value = self._parameter['value'][1]
                    xs = [0.0]
                    ys = [0.0]
                    self._parameter['value'] = ('FLOAT', float_value, xs, ys)
                elif self._parameter['value'][0] == 'CURVE':
                    float_value = self._parameter['default'][1]
                    xs = self._parameter['value'][1]
                    ys = self._parameter['value'][2]
                else:
                    raise AssertionError('Invalid float-curve type')  # pragma: nocover  This is a developer mistake
                self._parameter['value'] = (mode, float_value, xs, ys)

        # Add new version handlers here, or merge with the above group.

        if from_version != _LATEST_VERSION:
            raise AssertionError('New version added without migration code.')  # pragma: nocover


class Group:
    """
    A class representing a group of parameters.

    `Group` is typically used to represent a feature type. For example, if arcs
    can represent either shorelines or bridge bases, there might be one group
    containing the parameters needed by riverbed arcs, and another group for
    the parameters needed by bridge base arcs. A particular arc might then have
    the riverbed group set to active to identify it as a riverbed arc.

    `Group` instances should not be created directly. See
    `Section.add_group()` and `Section.duplicate_group()` for alternatives
    meant for external use.
    """
    def __init__(self, section_name, group_name, group):
        """
        Initialize a Group.

        This is mainly for internal use by `Section`. See
        `Section.add_group()` and `Section.duplicate_group()` for
        alternatives meant for external use.

        Args:
            section_name: Name of the section containing the group.
            group_name: Name of the group.
            group: A dictionary generated by `Group.create()`.
        """
        self.section_name = section_name
        self.group_name = group_name
        self._group = group

    @staticmethod
    def create(
        label: str,
        is_active: bool | None = None,
        legal_on_interior: bool | None = None,
        correlation: str | None = None,
        description: str | None = None,
        is_default: bool | None = None,
    ):
        """
        Create a new dictionary suitable for passing to `self.__init__()`.

        This is mainly for internal use by `Section`. See
        `Section.add_group()` and `Section.duplicate_group()` for
        alternatives meant for external use.

        Args:
            label: User-friendly label for the parameter.
            is_active: Whether the group is active.
            legal_on_interior: Whether the group is legal on interior features.
            correlation: Name of a global group this one correlates with. This
                is mainly used by GMI, and even then only so it can preserve
                information from a .2dm file.
            description: A short piece of text providing extra detail about
                the group. Typically one or two sentences.
            is_default: Whether this is the default group.
        """
        value: dict = {'parameters': {}, 'label': label}
        if is_active is not None:
            value['is_active'] = is_active
        if legal_on_interior is not None:
            value['legal_on_interior'] = legal_on_interior
        if correlation is not None:
            value['correlation'] = correlation
        if description is not None:
            value['description'] = description
        if is_default is not None:
            value['is_default'] = is_default
        return value

    def parameter(self, name: GmiName) -> Parameter:
        """
        Get a parameter in this group.

        Args:
            name: Name of the parameter to get.

        Returns:
            The parameter.
        """
        return Parameter(self.section_name, self.group_name, name, self._group['parameters'][name])

    @property
    def parameter_names(self) -> Iterable[GmiName]:
        """The names of all the parameters in this group."""
        return self._group['parameters'].keys()

    @property
    def is_active(self) -> bool:
        """Whether the group is active."""
        return self._group.get('is_active', False)

    @is_active.setter
    def is_active(self, value: bool):
        """
        Whether the group is active.

        Args:
            value: The new value.
        """
        self._group['is_active'] = value

    @property
    def is_default(self) -> bool:
        """Whether the group is the default group."""
        return self._group.get('is_default', False)

    @property
    def legal_on_interior(self) -> bool:
        """Whether a boundary condition group can be used on interior features."""
        return self._group['legal_on_interior']

    @property
    def correlation(self) -> str:
        """
        The name of the group this is correlated with.

        Returns:
            The correlated group, or empty for no correlation.
        """
        if 'correlation' in self._group and self._group['correlation'] is not None:
            return self._group['correlation']
        return ''

    @property
    def label(self) -> str:
        """The group's label."""
        return self._group['label']

    @label.setter
    def label(self, value):
        """The group's label."""
        self._group['label'] = value

    @property
    def description(self) -> str:
        """The description text, if there is one, or an empty string if there isn't."""
        if 'description' in self._group and self._group['description'] is not None:
            return self._group['description']
        return ''

    def add_boolean(self, name: GmiName, label: str, default: bool, description: str = '') -> Parameter:
        """
        Add a boolean variable.

        Args:
            name: Name to identify the parameter by in code. Must be unique.
            label: Name to identify the parameter by to the user.
            default: Default value for the parameter.
            description: A short piece of text providing extra detail about
                the parameter. Typically one or two sentences.

        Returns:
            The newly created parameter.
        """
        self._group['parameters'][name] = Parameter.create(Type.BOOLEAN, label, default, description)
        return Parameter(self.section_name, self.group_name, name, self._group['parameters'][name])

    def add_checkbox_curve(
        self,
        name: GmiName,
        label: str,
        checkbox_label: str,
        axis_titles: list[str],
        modes: tuple[Curve, Curve],
        default_check_state: bool,
        default_constant: float = 0.0,
        low: float = MIN_FLOAT,
        high: float = MAX_FLOAT,
        description: str = '',
        checkbox_description: str = ''
    ) -> Parameter:
        """
        Add a curve variable.

        Args:
            name: Name to identify the parameter by in code. Must be unique.
            label: Name to identify the parameter by to the user.
            checkbox_label: Label to put on the checkbox.
            axis_titles: Titles for the axes.
            default_check_state: True if the checkbox is checked by default, else False.
            default_constant: Default value for when in constant mode.
            low: Minimum allowed value when in constant mode.
            high: Maximum allowed value when in constant mode.
            modes: Mapping from mode name to a `Curve` instance describing how
                the curve should behave in that mode. Mode names should be
                human-readable labels.
            description: A short piece of text providing extra detail about
                the parameter. Typically one or two sentences.
            checkbox_description: Description to put on the checkbox.

        Returns:
            The newly created parameter.
        """
        self._group['parameters'][name] = Parameter.create(
            Type.CHECKBOX_CURVE, label, (default_check_state, default_constant), description
        )
        parameter = Parameter(self.section_name, self.group_name, name, self._group['parameters'][name])
        parameter.checkbox_label = checkbox_label
        if checkbox_description:
            parameter.checkbox_description = checkbox_description
        parameter.low = low
        parameter.high = high
        parameter.axis_titles = axis_titles
        parameter.modes = modes
        return parameter

    def add_dataset(self, name: GmiName, label: str, description: str = '') -> Parameter:
        """
        Add a dataset variable.

        Args:
            name: Name to identify the parameter by in code. Must be unique.
            label: Name to identify the parameter by to the user.
            description: A short piece of text providing extra detail about
                the parameter. Typically one or two sentences.

        Returns:
            The newly created parameter.
        """
        self._group['parameters'][name] = Parameter.create(Type.DATASET, label, '', description)
        parameter = Parameter(self.section_name, self.group_name, name, self._group['parameters'][name])
        return parameter

    def add_float(
        self,
        name: GmiName,
        label: str,
        default: float,
        low: float = MIN_FLOAT,
        high: float = MAX_FLOAT,
        description: str = ''
    ) -> Parameter:
        """
        Add a float variable.

        Args:
            name: Name to identify the parameter by in code. Must be unique.
            label: Name to identify the parameter by to the user.
            default: Default value for the parameter.
            low: Minimum allowed value.
            high: Maximum allowed value.
            description: A short piece of text providing extra detail about
                the parameter. Typically one or two sentences.

        Returns:
            The newly created parameter.
        """
        self._group['parameters'][name] = Parameter.create(Type.FLOAT, label, default, description)
        parameter = Parameter(self.section_name, self.group_name, name, self._group['parameters'][name])
        parameter.high = high
        parameter.low = low
        return parameter

    def add_date_time(
        self,
        name: GmiName,
        label: str,
        default: datetime | None,  # None -> Jan 1, 2000
        description: str = '',
        calendar_popup: bool = True
    ) -> Parameter:
        """
        Add a datetime variable.

        Args:
            name: Name to identify the parameter by in code. Must be unique.
            label: Name to identify the parameter by to the user.
            default: Default value for the parameter. If None, datetime(year=2000, month=1, day=1) is used.
            description: A short piece of text providing extra detail about
                the parameter. Typically one or two sentences.
            calendar_popup: True to display an arrow on the right of the edit field, which opens a calendar widget.

        Returns:
            The newly created parameter.
        """
        if default is None:
            default = datetime(year=2000, month=1, day=1)
        default_iso_string = default.isoformat()
        self._group['parameters'][name] = Parameter.create(Type.DATE_TIME, label, default_iso_string, description)
        parameter = Parameter(self.section_name, self.group_name, name, self._group['parameters'][name])
        parameter.calendar_popup = calendar_popup
        return parameter

    def add_float_or_xy_series(
        self,
        name: GmiName,
        label: str,
        axis_titles: list[str],
        default_value: float,
        low: float = MIN_FLOAT,
        high: float = MAX_FLOAT,
        default_mode: str = "FLOAT",
        use_dates: bool = False,
        description: str = ''
    ) -> Parameter:
        """
        Add a variable that can be either a float or a curve.

        Similar to add_float_curve, except this only stores the curve's ID.

        Args:
            name: Name to identify the parameter by in code. Must be unique.
            label: Name to identify the parameter by to the user.
            axis_titles: Titles for the axes.
            default_value: Default value for when in FLOAT mode.
            low: Minimum value when in FLOAT mode.
            high: Maximum value when in FLOAT mode.
            default_mode: Default mode (`'FLOAT'` or `'CURVE'`)
            use_dates: flag to specify that the x column of the time series should be a date time.
            description: A short piece of text providing extra detail about
                the parameter. Typically one or two sentences.

        Returns:
            The newly created parameter.
        """
        self._group['parameters'][name] = Parameter.create(
            Type.FLOAT_OR_CURVE, label, [default_mode, default_value], description
        )
        parameter = Parameter(self.section_name, self.group_name, name, self._group['parameters'][name])
        parameter.low = low
        parameter.high = high
        parameter.axis_titles = axis_titles
        parameter.use_dates = use_dates
        return parameter

    def add_float_curve(
        self,
        name: GmiName,
        label: str,
        axis_titles: list[str],
        default_value: float,
        low: float = MIN_FLOAT,
        high: float = MAX_FLOAT,
        default_mode: str = "FLOAT",
        use_dates: bool = False,
        description: str = ''
    ) -> Parameter:
        """
        Add a float/series variable.

        Similar to add_float_or_xy_series, except this stores the curve's data. The default curve always has a
        single point at (0.0, 0.0).

        The value is a tuple of (mode, float_value, xs, ys). mode is either 'FLOAT' or 'CURVE', float_value is the
        current value for float mode, and xs and ys are the x and y values for curve mode.

        Args:
            name: Name to identify the parameter by in code. Must be unique.
            label: Name to identify the parameter by to the user.
            axis_titles: Titles for the axes.
            default_value: Default value for when in FLOAT mode.
            low: Minimum value when in FLOAT mode.
            high: Maximum value when in FLOAT mode.
            default_mode: Default mode (`'FLOAT'` or `'CURVE'`)
            use_dates: flag to specify that the x column of the time series should be a date time.
            description: A short piece of text providing extra detail about
                the parameter. Typically one or two sentences.

        Returns:
            The newly created parameter.
        """
        default = (default_mode, default_value, [0.0], [0.0])
        self._group['parameters'][name] = Parameter.create(Type.FLOAT_CURVE, label, default, description)
        parameter = Parameter(self.section_name, self.group_name, name, self._group['parameters'][name])
        parameter.low = low
        parameter.high = high
        parameter.axis_titles = axis_titles
        parameter.use_dates = use_dates
        return parameter

    def add_input_file(
        self, name: GmiName, label: str, default: str, file_filter: str, description: str = ''
    ) -> Parameter:
        """
        Add an input file parameter.

        Args:
            name: Name to identify the parameter by in code. Must be unique.
            label: Name to identify the parameter by to the user.
            default: Default value for the parameter.
            file_filter: File filter for the parameter.
            description: A short piece of text providing extra detail about
                the parameter. Typically one or two sentences.

        Returns:
            The newly created parameter.
        """
        self._group['parameters'][name] = Parameter.create(Type.INPUT_FILE, label, default, description)
        parameter = Parameter(self.section_name, self.group_name, name, self._group['parameters'][name])
        parameter.file_filter = file_filter
        return parameter

    def add_input_folder(self, name: GmiName, label: str, default: str, description: str = '') -> Parameter:
        """
        Add an input folder parameter.

        Args:
            name: Name to identify the parameter by in code. Must be unique.
            label: Name to identify the parameter by to the user.
            default: Default value for the parameter.
            description: A short piece of text providing extra detail about
                the parameter. Typically one or two sentences.

        Returns:
            The newly created parameter.
        """
        self._group['parameters'][name] = Parameter.create(Type.INPUT_FOLDER, label, default, description)
        return Parameter(self.section_name, self.group_name, name, self._group['parameters'][name])

    def add_integer(
        self,
        name: GmiName,
        label: str,
        default: int,
        low: int = -2147483648,
        high: int = 2147483647,
        description: str = ''
    ) -> Parameter:
        """Add an integer variable.

        Args:
            name: Name to identify the parameter by in code. Must be unique.
            label: Name to identify the parameter by to the user.
            default: Default value for the parameter.
            low: Minimum allowed value.
            high: Maximum allowed value.
            description: A short piece of text providing extra detail about
                the parameter. Typically one or two sentences.

        Returns:
            The newly created parameter.
        """
        self._group['parameters'][name] = Parameter.create(Type.INTEGER, label, default, description)
        parameter = Parameter(self.section_name, self.group_name, name, self._group['parameters'][name])
        parameter.low = low
        parameter.high = high
        return parameter

    def add_option(
        self,
        name: GmiName,
        label: str,
        default: str,
        options: Optional[list] = None,
        description: str = ''
    ) -> Parameter:
        """Add an option variable.

        Use set_options to set the allowed options.

        Args:
            name: Name to identify the parameter by in code. Must be unique.
            label: Name to identify the parameter by to the user.
            default: Default value for the parameter.
            options: List of strings that are the options for the parameter
            description: A short piece of text providing extra detail about
                the parameter. Typically one or two sentences.

        Returns:
            The newly created parameter.
        """
        self._group['parameters'][name] = Parameter.create(Type.OPTION, label, default, description)
        par = Parameter(self.section_name, self.group_name, name, self._group['parameters'][name])
        if options is not None:
            par.options = options
        return par

    def add_output_file(
        self, name: GmiName, label: str, default: str, file_filter: str, description: str = ''
    ) -> Parameter:
        """
        Add an output file parameter.

        Args:
            name: Name to identify the parameter by in code. Must be unique.
            label: Name to identify the parameter by to the user.
            default: Default value for the parameter.
            file_filter: File filter for the parameter.
            description: A short piece of text providing extra detail about
                the parameter. Typically one or two sentences.

        Returns:
            The newly created parameter.
        """
        self._group['parameters'][name] = Parameter.create(Type.OUTPUT_FILE, label, default, description)
        parameter = Parameter(self.section_name, self.group_name, name, self._group['parameters'][name])
        parameter.file_filter = file_filter
        return parameter

    def add_text(
        self, name: GmiName, label: str, default: str = '', description: str = '', required: bool = False
    ) -> Parameter:
        """
        Add a text variable.

        Args:
            name: Name to identify the parameter by in code. Must be unique.
            label: Name to identify the parameter by to the user.
            default: Default value for the parameter.
            description: A short piece of text providing extra detail about
                the parameter. Typically one or two sentences.
            required (bool): If True, a value for the text must be supplied in the GroupSetDialog.

        Returns:
            The newly created parameter.
        """
        self._group['parameters'][name] = Parameter.create(Type.TEXT, label, default, description, required)
        return Parameter(self.section_name, self.group_name, name, self._group['parameters'][name])

    def add_xy_series(
        self,
        name: GmiName,
        label: str,
        axis_titles: list[str],
        default: int = -1,
        use_dates: bool = False,
        description: str = ''
    ) -> Parameter:
        """
        Add an xy-series variable.

        This is similar to add_curve, except only the curve's ID is stored internally.

        Args:
            name: Name to identify the parameter by in code. Must be unique.
            label: Name to identify the parameter by to the user.
            axis_titles: Titles for the X and Y axes.
            default: Default value for the parameter.
            use_dates: flag to specify that the x column of the times series should be a date time
            description: A short piece of text providing extra detail about
                the parameter. Typically one or two sentences.

        Returns:
            The newly created parameter.
        """
        self._group['parameters'][name] = Parameter.create(Type.SERIES, label, default, description)
        parameter = Parameter(self.section_name, self.group_name, name, self._group['parameters'][name])
        parameter.axis_titles = axis_titles
        parameter.use_dates = use_dates
        return parameter

    def add_curve(
        self,
        name: GmiName,
        label: str,
        axis_titles: list[str],
        use_dates: bool = False,
        description: str = ''
    ) -> Parameter:
        """
        Add an xy-series variable.

        This is similar to add_xy_series, except the series data is stored internally. The default curve always has a
        single point at (0.0, 0.0).

        Args:
            name: Name to identify the parameter by in code. Must be unique.
            label: Name to identify the parameter by to the user.
            axis_titles: Titles for the X and Y axes.
            use_dates: flag to specify that the x column of the times series should be a date time
            description: A short piece of text providing extra detail about
                the parameter. Typically one or two sentences.

        Returns:
            The newly created parameter.
        """
        default = [0.0], [0.0]
        self._group['parameters'][name] = Parameter.create(Type.CURVE, label, default, description)
        parameter = Parameter(self.section_name, self.group_name, name, self._group['parameters'][name])
        parameter.axis_titles = axis_titles
        parameter.use_dates = use_dates
        return parameter

    def add_table(
        self,
        name: GmiName,
        label: str,
        default,
        table_definition: TableDefinition,
        description: str = '',
        display_table: bool = False,
    ) -> Parameter:
        """
        Add a table variable.

        Args:
            name: Name to identify the parameter by in code. Must be unique.
            label: Name to identify the parameter by to the user.
            default: Default values for the table as a 2D list.
            table_definition (TableDefinition): Defines the table column types.
            description: A short piece of text providing extra detail about
                the parameter. Typically one or two sentences.
            display_table: If True, the table is displayed. If false, a table dialog is opened via a button.

        Returns:
            The newly created parameter.
        """
        self._group['parameters'][name] = Parameter.create(Type.TABLE, label, default=default, description=description)
        parameter = Parameter(self.section_name, self.group_name, name, self._group['parameters'][name])
        parameter.table_definition = table_definition
        parameter.display_table = display_table
        return parameter

    def compatible(self, other: Group) -> bool:
        """
        Check compatibility.

        Two groups are considered compatible if they contain the same names,
        and each parameter in one group is compatible with the parameter in the
        other group with the same name.

        Args:
            other: Another instance to test compatibility against.

        Returns:
            Whether self and other are compatible.
        """
        if self.parameter_names != other.parameter_names:
            return False
        for parameter_name in self.parameter_names:
            if not self.parameter(parameter_name).compatible(other.parameter(parameter_name)):
                return False
        return True

    def extract_values(self, only_group_activity: bool = False, include_default: bool = True) -> dict:
        """
        Extract all the values out of the `Group`.

        This is meant for internal use. External code should typically extract
        values at the `Section` level.

        Args:
            only_group_activity: Whether to extract only group activity.
            include_default: Controls how a Parameter's value is extracted when
                it has no value. If True, the Parameter's default value is
                extracted. Otherwise, no value is extracted.

                True preserves the current value of a parameter, even if the
                current value is a default one, and the default changes later.
                This ensures that changing the default doesn't change the
                behavior of existing projects.

                False preserves whether the parameter has ever been assigned,
                ensuring that Parameter.is_default produces the same result
                after restoring that it did before extracting.

        Return:
            dict representing all the values for the Group.
        """
        parameters = {}

        for name in self.parameter_names:
            parameter = self.parameter(name)
            if include_default or parameter.has_value:
                value = self.parameter(name).extract_value()
                parameters[name] = value

        values_dict: dict[str, Any] = {}
        if parameters and not only_group_activity:
            values_dict['parameters'] = parameters
        if (is_active := self._group.get('is_active', None)) is not None:
            values_dict['is_active'] = is_active
        return values_dict

    def restore_values(self, values: dict):
        """
        Restore extracted values to the Group.

        This is meant for internal use. External code should typically restore
        values at the `Section` level.

        Args:
            values: dict representing the values in a Group.
        """
        if (is_active := values.get('is_active', None)) is not None:
            self.is_active = is_active
        for name in values.get('parameters', {}):
            if name in self.parameter_names:
                self.parameter(name).restore_value(values['parameters'][name])

    def clear_values(self):
        """Clear all values from the `Group`."""
        self._group.pop('is_active', None)
        for parameter_name in self.parameter_names:
            parameter = self.parameter(parameter_name)
            parameter.clear_value()

    def to_literal(self):
        """
        Convert the `Group` from an in-memory representation to a Python literal.

        This is typically called right before serialization.

        :meta private:  This is only public so `Section` can call it.
        """
        for parameter_name in self.parameter_names:
            self.parameter(parameter_name).to_literal()

    def from_literal(self):
        """
        Convert the `Group` from a Python literal to a more convenient in-memory representation.

        This is typically called right after deserialization.

        :meta private:  This is only public so `Section` can call it.
        """
        for parameter_name in self.parameter_names:
            self.parameter(parameter_name).from_literal()

    def migrate(self, from_version):
        """
        Migrate the internal structure of the `Group` to the latest version.

        External code shouldn't have to call this.

        Args:
            from_version: The version being migrated from.
        """
        # See GenericModel._migrate() for a suggested pattern for doing this if we ever need to do more than just
        # pass through to the groups.
        for parameter_name in self.parameter_names:
            self.parameter(parameter_name).migrate(from_version)


class Section:
    """
    A class representing a set of groups of parameters.

    A `Section` is typically used to represent all the possible types a
    particular feature might take on. There is generally one section for each
    of points, arcs, polygons, materials, and global variables.

    `Section` instances should not be created directly. See the various
    `GenericModel.*_parameters` fields for alternatives meant for external use.
    """
    def __init__(self, name: GmiName, groups: dict):
        """
        Initialize an empty `Section`.

        This is mainly for internal use by `GenericModel`. See the
        `GenericModel.*_parameters` fields for alternatives meant for external
        use.

        Args:
            name: Name of the `Section`.
            groups: Dictionary returned by `self.create()`.
        """
        self._group_set = groups
        self._name = name

    @staticmethod
    def create(exclusive_groups: bool = False):
        """
        Create a new dictionary suitable for passing to `self.__init__()`.

        This is mainly for internal use by `GenericModel`. See the
        `GenericModel.*_parameters` fields for alternatives meant for external
        use.

        Args:
            exclusive_groups: Whether groups are mutually exclusive. If True,
                then only one group should be active at a time.

                Note: This only provides metadata for external code. `Section`
                does not enforce it.
        """
        return {'exclusive_groups': exclusive_groups, 'groups': {}}

    def __eq__(self, other: Any):
        """
        Check whether this section is equal to another one.

        Unlike `self.compatible()`, this requires the sections to be identical.

        Args:
            other: The other section to compare to.

        Returns:
            True if the sections are equal.
        """
        return isinstance(other, Section) and self._group_set == other._group_set

    def __ne__(self, other: Any):
        """
        Check whether this section is unequal to another one.

        Unlike `self.compatible()`, this fails even if only values differ.

        Args:
            other: The other section to compare to.

        Returns:
            True if the sections are unequal.
        """
        return not (self == other)

    def copy(self) -> Section:
        """Get a copy of this section."""
        other = copy.deepcopy(self._group_set)
        return Section(self._name, other)

    def group(self, group_name: GmiName) -> Group:
        """
        Get a group.

        Args:
            group_name: Name of the group to get.
        """
        return Group(self.name, group_name, self._group_set['groups'][group_name])

    def active_group_name(self, value_for_none: Optional[GmiName] = None, value_for_multiple: Optional[GmiName] = None):
        """
        Get the name of the currently active group.

        Args:
            value_for_none: Value to return if no groups are active.
            value_for_multiple: Value to return if more than one group is active.
        """
        active_group_names = self.active_group_names
        if len(active_group_names) > 1:
            return value_for_multiple
        elif len(active_group_names) == 1:
            return active_group_names[0]

        return value_for_none

    @property
    def active_group_names(self) -> list[GmiName]:
        """The names of all the active groups."""
        names = [name for name in self.group_names if self.group(name).is_active]
        if not names and self.default_group_name:
            return [self.default_group_name]
        return names

    def deactivate_groups(self):
        """
        Set all groups to inactive.

        After this returns, `self.group(name).is_active` will be `False` for
        every name in `self.group_names`.
        """
        for group_name in self.group_names:
            self.group(group_name).is_active = False

    @property
    def default_group_name(self) -> Optional[GmiName]:
        """The name of the default group, which is active when no other group is."""
        for group_name in self.group_names:
            group = self.group(group_name)
            if group.is_default:
                return group_name
        return None

    def has_group(self, group_name: GmiName) -> bool:
        """
        Check whether a group name exists in this section.

        Args:
            group_name: Name of the group to check for.

        Returns:
            Whether a group with the given name currently exists.
        """
        return group_name in self._group_set['groups']

    @property
    def exclusive_groups(self) -> bool:
        """Whether groups are mutually exclusive."""
        return self._group_set['exclusive_groups']

    @exclusive_groups.setter
    def exclusive_groups(self, value: bool):
        """Whether groups are mutually exclusive."""
        self._group_set['exclusive_groups'] = value

    @property
    def parameters(self) -> Iterable[tuple[GmiName, GmiName, Parameter]]:
        """
        An iterable of all the parameters contained in this `Section`.

        Returns:
            Iterable of tuples of (group_name, parameter_name, parameter).
        """
        for group_name in self.group_names:
            group = self.group(group_name)
            for parameter_name in group.parameter_names:
                parameter = group.parameter(parameter_name)
                yield group_name, parameter_name, parameter

    @property
    def group_names(self) -> Sequence[GmiName]:
        """
        Get the names of all the groups in the section.

        Returns:
            A list of names in the section.
        """
        return list(self._group_set['groups'].keys())

    @property
    def groups(self) -> Iterable[tuple[GmiName, Group]]:
        """An iterable of `(group_name, group)` for each group in the `Section`."""
        for group_name in self.group_names:
            yield group_name, self.group(group_name)

    @property
    def name(self) -> str:
        """
        The section's name, e.g. 'global', 'points', 'materials', etc.

        Note that these names are an implementation detail. This is mainly
        to support `Parameter.add_dependency()`'s list option, which is
        deprecated.
        """
        return self._name

    def add_group(
        self,
        group_name: GmiName,
        label: str,
        is_active: bool | None = None,
        allow_interior: Optional[bool] = None,
        correlation: Optional[GmiName] = None,
        description: Optional[str] = None,
        is_default: Optional[bool] = None,
    ):
        """
        Add a group.

        Args:
            group_name: Name to identify the group by in code. Must be unique.
            label: Name to identify the group by to the user.
            is_active: Whether the group is active.
            allow_interior: Whether this group should be allowed on features
                that are not on the boundary of the domain. This likely doesn't
                make sense for global, model, and material parameters, since
                they don't have a location. It may not make sense for anything
                else either, since a point can be in a coverage in multiple
                simulations with multiple meshes, making it a boundary node in
                one simulation but not in another. The current GMI only keeps
                this so that it can write the old information back out again.
            correlation: Name of a group in the global parameters which the
                new group is correlated with. The old GMI used this to make
                some point types unavailable depending on which global groups
                were active, but this no longer works since a point might be in
                a coverage that is part of multiple simulations, with
                conflicting activity for the correlated group. The current GMI
                only uses this to preserve the data so it can be written back
                to the file again.
            description: A short piece of text providing extra detail about
                the group. Typically one or two sentences.
            is_default: Whether this is the default group. The default group is
                considered active when no other group is.
        """
        self._group_set['groups'][group_name] = Group.create(
            label, is_active, allow_interior, correlation, description, is_default
        )
        return Group(self.name, group_name, self._group_set['groups'][group_name])

    def duplicate_group(self, existing_name: GmiName, new_name: GmiName, new_label: str) -> Group:
        """
        Add a new group based on an existing one.

        Args:
            existing_name: Name of the group to copy.
            new_name: Name to assign the new group.
            new_label: Label to assign the new group.

        Returns:
            The newly created group.
        """
        existing_group = self._group_set['groups'][existing_name]
        new_group = copy.deepcopy(existing_group)
        self._group_set['groups'][new_name] = new_group
        group = Group(self.name, new_name, new_group)
        group.label = new_label
        return group

    def remove_group(self, name: GmiName):
        """
        Remove an existing group.

        Args:
            name: Name of the group to remove.
        """
        del self._group_set['groups'][name]

    def to_pretty_string(self) -> str:
        """
        Convert to a string suitable for storing, that is also human-readable.

        The output will be a human-friendly string representation of the Section. Unlike GenericModel, Section has no
        constructor that accepts the resulting string as input. This is mainly useful for producing values that can be
        compared against baselines, or for debugging whether a section looks reasonable.
        """
        return pretty_print(self._group_set)

    def compatible(self, other: Section) -> bool:
        """
        Checks for compatibility between this `Section` and `other`.

        Two sections are considered compatible if they contain the same names,
        and each group in one section is compatible with the group in the
        other section with the same name.

        Args:
            other: `Section` to compare against.

        Returns:
            Whether they are compatible.
        """
        if self.name == 'materials':
            self_unassigned = self.group(UNASSIGNED_MATERIAL_ID)
            other_unassigned = other.group(UNASSIGNED_MATERIAL_ID)
            return self_unassigned.compatible(other_unassigned)

        if self.group_names != other.group_names:
            return False

        for group_name in self.group_names:
            if not self.group(group_name).compatible(other.group(group_name)):
                return False

        return True

    def extract_values(self, include_default: bool = True) -> str:
        """
        Extract all the values out of the `Section`.

        The return value can later be passed to `self.restore_values()` to
        return the `Section` to the state it was extracted from.

        The extracted values include group activity. Call
        `self.deactivate_groups()` first to remove that information from the
        `Section` if it is undesired.

        Args:
            include_default: Controls how a Parameter's value is extracted when
                it has no value. If True, the Parameter's default value is
                extracted. Otherwise, no value is extracted.

                True preserves the current value of a parameter, even if the
                current value is a default one, and the default changes later.
                This ensures that changing the default doesn't change the
                behavior of existing projects.

                False preserves whether the parameter has ever been assigned,
                ensuring that Parameter.is_default produces the same result
                after restoring that it did before extracting.

        Return:
            A string representation of the values in the `Section`.
        """
        values = {}
        for group_name in self.group_names:
            value = self.group(group_name).extract_values(include_default=include_default)
            if value:
                values[group_name] = value

        return _to_string(values)

    def restore_values(self, values: str):
        """
        Replace all values in the `Section` with the provided ones.

        Any parameter not set by something in `values` will have its value
        (but not its definition) reset to its default. This is equivalent to
        calling `self.clear_values()` before restoring. In most cases this
        won't matter since `values` normally contains a value for every
        parameter, but it may be significant if a new parameter is added.

        If `values` is something returned by `self.extract_values()`, then this
        will restore the `Section` to its state when `self.extract_values()`
        was called.

        Args:
            values: Something returned by `self.extract_values()`.
        """
        self.clear_values()
        self._update(values)

    def extract_group_activity(self) -> str:
        """
        Extract the group activity from the section.

        This is mainly useful for materials, where each polygon can have a
        different material active, but they should all have the same values
        for the materials. The values particular to a polygon can be extracted
        with this method, then `self.deactivate_groups()` can be used to erase
        this information, and finally `self.extract_values()` can be used to
        extract the information common to all material polygons.

        The inverse would be to use `self.restore_values()` with the values
        common to all polygons, then `self.restore_group_activity()` to restore
        the information particular to this polygon.
        """
        values = {}
        for group_name in self.group_names:
            value = self.group(group_name).extract_values(only_group_activity=True)
            if value:
                values[group_name] = value

        return _to_string(values)

    def restore_group_activity(self, activity: str):
        """
        Restore group activity into the `Section`.

        This method resets all groups to their default values, then assigns
        group activity from the input, similar to how `self.restore_values()`
        behaves.

        Args:
            activity: Something returned by `self.extract_group_activity()`
        """
        self.deactivate_groups()
        self._update(activity)

    def _update(self, values: str):
        """
        Replace values in the Section with the provided ones.

        Parameters not present in values will retain their original values.

        Args:
            values: Something returned by self.extract_values.
        """
        values_dict = _from_string(values)
        for group_name in values_dict:
            if group_name in self.group_names:
                self.group(group_name).restore_values(values_dict[group_name])

    def clear_values(self):
        """Clear all values from the Section."""
        for group_name in self.group_names:
            group = self.group(group_name)
            group.clear_values()

    def to_literal(self):
        """
        Convert the `Section` from an in-memory representation to a Python literal.

        This is typically called right before serialization.

        :meta private:  This is only public so `GenericModel` can call it.
        """
        for group_name in self.group_names:
            self.group(group_name).to_literal()

    def from_literal(self):
        """
        Convert the `Section` from a Python literal to a more convenient in-memory representation.

        This is typically called right after deserialization.

        :meta private:  This is only public so `GenericModel` can call it.
        """
        for group_name in self.group_names:
            self.group(group_name).from_literal()

    def migrate(self, from_version: int):
        """
        Migrate the Section to the latest version.

        External code shouldn't have to call this.

        Args:
            from_version: The version being migrated from.
        """
        # See GenericModel._migrate() for a suggested pattern for doing this if we ever need to do more than just
        # pass through to the groups.
        for group_name in self.group_names:
            self.group(group_name).migrate(from_version)


# Section used to be called GroupSet.
GroupSet = Section


class GenericModel:
    """
    A class representing a generic model.

    `GenericModel` allows defining a model at runtime.

    A `GenericModel` is organized into several `Sections`, each of which contains
    all the parameters that can be assigned to a particular feature type (points,
    arcs, polygons). It also has sections for materials and global parameters.
    The latter are useful for model control data that applies to everything. The
    special model section is mainly for `xmsgmi` internal use.

    Each `Section` is itself organized into several `Groups`, which contain related
    parameters. Groups can be made active or inactive. In the global section, groups
    might be organized based on features offered by the model (e.g. one group for
    things about wind speed, another for things about evaporation rates). In the
    point, arc, and polygon sections, groups are typically used to represent the
    different types a feature might take on (e.g. a particular point might represent
    either a source or sink, so there might be a 'Source' group and a 'Sink' group,
    and whether the point is a Source or Sink is determined by which group is
    active). The material section generally has one group per material, with all the
    groups' structures being the same, and only varying based on value.

    Each `Group` then contains `Parameters`, which define things like names and
    types for each variable. A `Parameter` can also have a value assigned to it.

    With the exception of global parameters, it is common to have multiple instances
    of the model (one for each point, for example). The entire model, or just a
    section, can be copied with its `copy()` member. The current values can also
    be extracted and restored with `Section.extract_values()` and
    `Section.restore_values()`, respectively. The latter method is the currently
    used convention for handling multiple instances. It produces strings, which
    are usually easy to serialize.

    The model itself can be serialized by `GenericModel.to_string()` and restored
    using the `definitions` parameter on its constructor. A template with no
    values can be obtained with `GenericModel.to_template()`. Chaining these as
    `GenericModel.to_template().to_string()` is the easiest way to serialize a
    template.

    It is often helpful to compare two `GenericModel` instances while debugging
    or testing. `GenericModel.to_pretty_string()` provides a convenient way to
    do this. The resulting string is formatted with whitespace for readability,
    and all keys are sorted to ensure a stable order.
    """
    def __init__(
        self,
        exclusive_point_conditions=False,
        exclusive_arc_conditions=False,
        exclusive_polygon_conditions=False,
        exclusive_material_conditions=False,
        definitions: str | None = None
    ):
        """
        Initialize a generic model.

        Args:
            exclusive_point_conditions: Whether points should be restricted to
                a single type each.
            exclusive_arc_conditions: Whether arcs should be restricted to a
                single type each.
            exclusive_polygon_conditions: Whether polygons should be restricted
                to a single type each.
            exclusive_material_conditions: Whether polygons should be
                restricted to a single material each.
            definitions: A string returned by `self.to_string()`. If not None,
                the new `GenericModel` will be equivalent to the one that
                produced the string, and the other parameters will be ignored.
        """
        if definitions:
            self._definitions = _from_string(definitions)
            self._migrate()
            self.model_parameters.from_literal()
            self.global_parameters.from_literal()
            self.point_parameters.from_literal()
            self.arc_parameters.from_literal()
            self.polygon_parameters.from_literal()
            self.material_parameters.from_literal()
            return

        self._definitions = {
            'sections':
                {
                    'model_definitions': Section.create(),
                    'global_definitions': Section.create(),
                    'point_definitions': Section.create(exclusive_point_conditions),
                    'arc_definitions': Section.create(exclusive_arc_conditions),
                    'polygon_definitions': Section.create(exclusive_polygon_conditions),
                    'material_definitions': Section.create(exclusive_material_conditions),
                },
            'version': _LATEST_VERSION,
        }

    def __eq__(self, other: Any):
        """
        Check whether this model is equal to another one.

        Unlike `self.compatible()`, this requires the models to be identical.

        Args:
            other: The other model to compare to.

        Returns:
            True if the models are equal.
        """
        return isinstance(other, GenericModel) and self._definitions == other._definitions

    def __ne__(self, other: Any) -> bool:
        """
        Check whether this model is unequal to another one.

        Unlike `self.compatible()`, this fails even if only values differ.

        Args:
            other: The other model to compare to.

        Returns:
            True if the models are unequal.
        """
        return not (self == other)

    def copy(self) -> GenericModel:
        """Get a copy of this model."""
        return GenericModel(definitions=self.to_string())

    @property
    def global_parameters(self) -> Section:
        """The global parameters."""
        return Section('global', self._definitions['sections']['global_definitions'])

    @property
    def point_parameters(self) -> Section:
        """The point parameters."""
        return Section('points', self._definitions['sections']['point_definitions'])

    @property
    def arc_parameters(self) -> Section:
        """The arc parameters."""
        return Section('arcs', self._definitions['sections']['arc_definitions'])

    @property
    def polygon_parameters(self) -> Section:
        """The polygon parameters."""
        return Section('polygons', self._definitions['sections']['polygon_definitions'])

    @property
    def material_parameters(self) -> Section:
        """The material parameters."""
        return Section('materials', self._definitions['sections']['material_definitions'])

    @material_parameters.setter
    def material_parameters(self, value: Section):
        """Set the material parameters."""
        self._definitions['sections']['material_definitions'] = value._group_set

    @property
    def model_parameters(self) -> Section:
        """
        The parameters describing the model.

        These are probably only useful for GMI itself.
        """
        return Section('model', self._definitions['sections']['model_definitions'])

    def _clear_all_values(self):
        """Clears all the values in the model."""
        self.global_parameters.clear_values()
        self.point_parameters.clear_values()
        self.arc_parameters.clear_values()
        self.polygon_parameters.clear_values()
        self.material_parameters.clear_values()
        self.model_parameters.clear_values()

    def to_template(self) -> GenericModel:
        """
        Get a copy of the model definition without any parameter values.

        Returns:
            A copy of the model.
        """
        gm_copy = copy.deepcopy(self)
        gm_copy._clear_all_values()
        return gm_copy

    def to_string(self) -> str:
        """
        Convert to a string suitable for storing.

        The returned string can be passed to `self.__init__()` as the
        `definitions` parameter to recreate this model.
        """
        self._to_literal()
        value = _to_string(self._definitions)
        self._from_literal()
        return value

    def to_pretty_string(self, include_version: bool = False) -> str:
        """
        Convert to a string suitable for storing, that is also human-readable.

        This is effectively the same as `self.to_string()`, with extra whitespace inserted to make it easier to read.
        It's most useful for testing and debugging.

        If include_version is True, it can also be passed to `self.__init__()`, same as the one from `self.to_string()`.

        Args:
            include_version: Whether to include the version of xmsgmi used to generate the model. If True, the output
                will include an extra value that makes it suitable for passing to GenericModel.__init__, but which makes
                the output change every time xmsgmi is updated. Typically, True should be used when creating an input
                string for a test, and False should be used when writing output to compare against a baseline.
        """
        definitions = copy.deepcopy(self._definitions)
        if not include_version:
            definitions.pop('version')

        return pretty_print(definitions)

    def compatible(self, other: GenericModel) -> bool:
        """
        Checks for compatibility between this `GenericModel` and `other`.

        This model is considered compatible with `other` if each of this one's
        `self.*_parameters` fields are compatible with the matching field in
        `other`.

        Args:
            other: GenericModel to compare against.

        Returns:
            Whether they are compatible.
        """
        if not self.model_parameters.compatible(other.model_parameters):
            return False
        if not self.global_parameters.compatible(other.global_parameters):
            return False
        if not self.point_parameters.compatible(other.point_parameters):
            return False
        if not self.arc_parameters.compatible(other.arc_parameters):
            return False
        if not self.polygon_parameters.compatible(other.polygon_parameters):
            return False
        if not self.material_parameters.compatible(other.material_parameters):
            return False

        return True

    def is_default(self) -> bool:
        """Check if this is a default-initialized model."""
        other = GenericModel()
        return self._definitions == other._definitions

    def section_from_target_type(self, target_type: TargetType) -> Section:
        """Returns `point_parameters`, `arc_parameters`, or `polygon_parameters` given the target type (feature type).

        Args:
            target_type: TargetType matching the desired `Section`.

        Returns:
            A `Section`.
        """
        if target_type == TargetType.point:
            section = self.point_parameters
        elif target_type == TargetType.arc:
            section = self.arc_parameters
        elif target_type == TargetType.polygon:
            section = self.polygon_parameters
        else:  # pragma: nocover
            raise ValueError('target_type')  # Added a new target type without adding support
        return section

    def _to_literal(self):
        """
        Convert the model from a convenient in-memory representation to a Python literal.

        This is typically called right before serialization.
        """
        self.model_parameters.to_literal()
        self.global_parameters.to_literal()
        self.point_parameters.to_literal()
        self.arc_parameters.to_literal()
        self.polygon_parameters.to_literal()
        self.material_parameters.to_literal()

    def _from_literal(self):
        """
        Convert the model from a Python literal to a more convenient in-memory representation.

        This is typically called right after deserialization.
        """
        self.model_parameters.from_literal()
        self.global_parameters.from_literal()
        self.point_parameters.from_literal()
        self.arc_parameters.from_literal()
        self.polygon_parameters.from_literal()
        self.material_parameters.from_literal()

    def _migrate(self):
        """Migrate the GenericModel's internal structures to the latest version."""
        starting_version = self._definitions.get('version', _PRE_VERSION_NUMBERS)

        # Versioning had some early weirdness that made life inconvenient. Versions were based on the version of
        # xmsgmi, which made them update automatically, but also meant that tests would break every time you update
        # xmsgmi. This block migrates the old automatic versions to the manually assigned ones we use now.
        if isinstance(starting_version, str):
            starting_version = parse_version(starting_version)
            if starting_version < parse_version('0.2.17'):
                starting_version = _ADDED_VERSION_NUMBERS
            else:
                starting_version = _REMOVED_VERSION_AUTO_UPDATE
        elif starting_version == -98:
            starting_version = _ADDED_VERSION_NUMBERS

        current_version = starting_version

        if current_version < _ADDED_VERSION_NUMBERS:
            self._definitions = {'sections': self._definitions, 'version': _LATEST_VERSION}
            current_version = _ADDED_VERSION_NUMBERS

        # Skip these since there are no changes at this level:
        # _RENAMED_CURVE_TO_SERIES_AND_ADDED_NEW_CURVE_TYPE
        # _REMOVED_VERSION_AUTO_UPDATE
        # _MADE_FLOAT_CURVE_FIXED_SIZE_VALUE

        if current_version < _MADE_FLOAT_CURVE_FIXED_SIZE_VALUE:
            current_version = _MADE_FLOAT_CURVE_FIXED_SIZE_VALUE

        # Add new version handlers here, or merge with the above group.

        if current_version != _LATEST_VERSION:
            raise AssertionError('New version added without migration code.')  # pragma: nocover

        self._definitions['version'] = current_version
        self.global_parameters.migrate(starting_version)
        self.model_parameters.migrate(starting_version)
        self.point_parameters.migrate(starting_version)
        self.arc_parameters.migrate(starting_version)
        self.polygon_parameters.migrate(starting_version)
        self.material_parameters.migrate(starting_version)


def _from_string(s: str) -> dict:
    """
    Convert a string produced by _to_string() back into the original dictionary.

    Args:
        s: The string to convert.

    Returns:
        The dictionary that produced the string.
    """
    # If this crashes, either you tried to restore something into a GenericModel that didn't come out of it, or you
    # put something bad into the model before serializing it or extracting from one of its sections.
    #
    # This string should be a valid Python literal, so sometimes, you can paste the contents of the string into a Python
    # REPL and it will tell you where the problem is.
    #
    # If you can set a breakpoint where you're extracting the string from the model, you can also call
    # GenericModel.to_pretty_string() and examine the output.
    #
    # Odds are, the problem will be a line that looks something like this: '<__main__.foo object at 0x000001D900B8B790>'
    # Which means whatever object that is, isn't allowed to be in the model.
    #
    # GenericModel can only contain objects of type str, bytes, int, bool, float, set, list, dict, tuple, and None.
    # Anything else needs to be converted to one of these before putting it in the model, and restored when taking it
    # out.
    try:
        return ast.literal_eval(s or '{}')
    except ValueError:
        raise ValueError('Unable to parse string: ' + repr(s))


def _to_string(d: dict) -> str:
    """
    Convert a dictionary as used by GenericModel and/or its contents into a string.

    Args:
        d: The dictionary to convert.

    Returns:
        A string representation of the dictionary.
    """
    value = repr(d)
    try:
        _from_string(value)
    except SyntaxError:
        raise ValueError('Generic Model contained values that are not a literal')
    return value


def _check_type(name, value, t):  # pragma: nocover
    """Function to check the type of something being assigned to a parameter."""
    try:
        _check_type_helper(value, t)
    except AssertionError:
        raise AssertionError(f'Invalid value for {name} of type {t}: {value}')


def _check_type_helper(value, t):  # pragma: nocover
    """Helper for _check_type."""
    if t == Type.BOOLEAN:
        assert isinstance(value, bool)
    elif t == Type.CHECKBOX_CURVE:
        assert isinstance(value[0], bool) or isinstance(value[0], int)  # TODO: Fix whoever sets to int
        assert isinstance(value[1], float) or isinstance(value[1], int)  # TODO: Fix whoever sets these inconsistently
    elif t in [
        Type.DATASET, Type.DATE_TIME, Type.INPUT_FILE, Type.INPUT_FOLDER, Type.OPTION, Type.OUTPUT_FILE, Type.TEXT
    ]:
        assert isinstance(value, str)
    elif t == Type.FLOAT:
        assert isinstance(value, float)
    elif t == Type.INPUT_FILE:
        assert isinstance(value, str)
    elif t == Type.INTEGER or t == Type.SERIES:
        assert isinstance(value, int)
    elif t == Type.CURVE:
        assert isinstance(value, tuple) and len(value) == 2
        assert len(value[0]) == len(value[1])
        for x, y in zip(value[0], value[1]):
            assert isinstance(x, float) and isinstance(y, float)
    elif t == Type.COLOR:
        assert isinstance(value, list) and len(value) == 3
        for c in value:
            assert isinstance(c, int) and 0 <= c <= 255
    elif t == Type.FLOAT_CURVE:
        assert isinstance(value, tuple) and len(value) == 4
        assert value[0] in ['FLOAT', 'CURVE']
        assert isinstance(value[1], float)
        assert isinstance(value[2], list) and isinstance(value[3], list) and len(value[2]) == len(value[3])
    elif t == Type.FLOAT_OR_CURVE:
        assert isinstance(value, list) and len(value) == 2
    elif t == Type.TABLE:
        assert isinstance(value, list)
    else:
        raise AssertionError('Unknown type')
