"""Function for pretty-printing a dictionary."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
from enum import Enum
from io import StringIO
from typing import Optional, TextIO

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules

_type_map = {str: 0, int: 1, bool: 2, float: 3, None: 4}


def _sorted(items):
    """
    Sort a sequence of items.

    Handles non-homogenous sequences.
    """
    items = [(_type_map[type(item)], item) for item in items]
    items.sort()
    return [item[1] for item in items]


def _is_simple_type(value: None | str | int | bool | float | dict | list | set | tuple | Enum):
    """
    Check whether something is a simple type that should be printed using repr().

    Args:
        value: The value to check.

    Returns:
        Whether value is a simple type.
    """
    if value is None or isinstance(value, str):
        return True
    if isinstance(value, int) or isinstance(value, bool) or isinstance(value, float):
        return True
    if isinstance(value, Enum):
        return False
    if value == {}:
        return True
    if isinstance(value, dict):
        return False

    for item in value:
        if isinstance(item, dict) or isinstance(item, Enum):
            return False

    return True


def _dict_helper(value, indent: str, out: TextIO):
    """
    Implementation of pretty_print() for dictionaries.

    Args:
        value: The value to print.
        indent: How much indentation to print.
        out: Where to write the output.
    """
    keys = _sorted(value.keys())

    out.write('{\n')
    child_indent = indent + '  '
    for key in keys:
        out.write(f'{child_indent}{repr(key)}: ')
        _helper(value[key], child_indent, out)
        out.write(',\n')
    out.write(f'{indent}}}')


def _sequence_helper(value, indent: str, out: TextIO):
    """
    Implementation of pretty_print() for lists.

    Args:
        value: The value to print.
        indent: How much indentation to print.
        out: Where to write the output.
    """
    start, end = '()' if isinstance(value, tuple) else '[]'
    if all(isinstance(item, Enum) for item in value):
        out.write(f"{start}'")
        out.write("', '".join(str(item) for item in value))
        out.write(f"'{end}")
        return

    out.write(f'{start}\n')
    child_indent = indent + '  '
    for item in value:
        out.write(child_indent)
        _helper(item, child_indent, out)
        out.write(',\n')
    out.write(f'{indent}{end}')


def _enum_helper(value, out: TextIO):
    """
    Implementation of pretty_print() for enums.

    Args:
        value: The value to print.
        out: Where to write the output.
    """
    out.write(f"'{str(value)}'")


def _helper(value, indent: str, out: TextIO):
    """
    Implementation of pretty_print().

    Args:
        value: The value to print.
        indent: How much indentation to print.
        out: Where to write the output.
    """
    if _is_simple_type(value):
        out.write(repr(value))
        return

    if isinstance(value, dict):
        _dict_helper(value, indent, out)
    elif isinstance(value, list) or isinstance(value, tuple):
        _sequence_helper(value, indent, out)
    elif isinstance(value, Enum):
        _enum_helper(value, out)
    else:  # pragma: no cover
        raise AssertionError('Unknown type')  # New type added without support for printing


def pretty_print(value, out: Optional[TextIO] = None):
    """
    Pretty-print a value.

    Similar to the built-in pprint, but has nicer formatting (at least for our uses).
    Also similar to yapf, but much faster.

    Args:
        value: The value to print.
        out: If supplied, string will be written to `out` rather than returned.

    Returns:
        If `out` is None, pretty-printed version of `value`. Otherwise, result is
        written to `out` and None is returned.
    """
    if out is None:
        out = StringIO()
        _helper(value, '', out)
        out.seek(0, 0)
        return out.read()
    else:
        _helper(value, '', out)
