r"""Writes the IN_HYD_LOCATION (.ihl) file, which describes hydrograph output locations.

I could not find any documentation of the file format on the `GSSHA wiki`_, but it's very simple:

.. _GSSHA wiki: https://www.gsshawiki.com/

::

    2
    3      34
    5      77

:code:`2` is the number of output hydgrographs.

:code:`3      34` means stream link 3, 34th node where "node" is a point on the polyline (see .cif file).
"""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import logging
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.gssha.data.bc_util import BcData
from xms.gssha.file_io.io_util import INT_WIDTH


def write(gssha_file_path: Path, stream_data: BcData) -> Path | None:
    """Writes the IN_HYD_LOCATION (.ihl) file and returns the file path.

    Args:
        gssha_file_path: .gssha file path.
        stream_data: Data about the streams.
    """
    if not stream_data or not stream_data.feature_bc:
        return None
    writer = InHydLocationFileWriter(gssha_file_path, stream_data)
    return writer.write()


class InHydLocationFileWriter:
    """Writes the IN_HYD_LOCATION (.ihl) file."""
    def __init__(self, gssha_file_path: Path, stream_data: BcData) -> None:
        """Initializes the class.

        Args:
            gssha_file_path (str | Path): .gssha file path.
            stream_data: Data about the streams.
        """
        super().__init__()
        self._ihl_file_path: Path = gssha_file_path.with_suffix('.ihl')
        self._stream_data = stream_data

        self._log = logging.getLogger('xms.gssha')
        self._nodes: set[int] = set()  # Set of nodes so we don't get duplicate link-node locations
        self._link_nodes: list[tuple[int, int]] = []  # List of tuple(link number, GSSHA node number)

    def write(self) -> Path | None:
        """Writes the IN_HYD_LOCATION (.ihl) file and returns the file path."""
        self._log.info('Writing .ihl file...')
        self._find_hydrograph_link_nodes()
        with open(self._ihl_file_path, 'w') as file:
            file.write(f'{len(self._link_nodes)}\n')
            for link, node in self._link_nodes:
                file.write(f'{link:<{INT_WIDTH}} {node:<{INT_WIDTH}}\n')
        return self._ihl_file_path

    def _find_hydrograph_link_nodes(self) -> None:
        """Searches the streams for where hydrographs are to be output and saves locations to a list."""
        for arc, gmi_group in self._stream_data.feature_bc.items():
            if gmi_group.parameter('hydrograph_down').value:
                # Make sure the node is only handled once by using the _nodes set
                feature_arc = self._stream_data.feature_from_id(arc[0], arc[1])
                if feature_arc.end_node not in self._nodes:
                    self._nodes.add(feature_arc.end_node)
                    link = self._stream_data.arc_links[feature_arc.id]
                    gssha_node = len(feature_arc.geometry.coords) - 1  # The number of the point on the arc
                    self._link_nodes.append((link, gssha_node))
