"""Functions to create the help url cache file."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path
import sys
import time
from urllib.error import URLError
import urllib.request

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy import file_io_util
from xms.guipy.__version__ import version as guipy_version

_test_fetch_time = 0  # When testing, the time we want it to take to fetch the wiki page


def script_filepath() -> str:
    """Return the full path to this .py file."""
    return os.path.abspath(__file__)


def _test_sleep(start_time: float) -> None:
    """When testing, sleep the amount of time left to reach _test_fetch_time.

    Args:
        start_time: The time right before calling _fetch_wiki_page()
    """
    # global _test_fetch_time
    if _test_fetch_time > 0:
        end_time = time.perf_counter()
        sleep_time = _test_fetch_time - (end_time - start_time)
        if sleep_time > 0:
            time.sleep(sleep_time)


def _create_url_cache(cache_path: Path, dialog_help_url: str) -> None:
    """Create the URL cache file.

    Args:
        cache_path: Where to save the cache to.
        dialog_help_url: Wiki page with all help urls, e.g. 'https://www.xmswiki.com/wiki/GMS:GMS_10.5_Dialog_Help'.
    """
    start_time = time.perf_counter()
    content = _fetch_wiki_page(dialog_help_url)
    _test_sleep(start_time)

    if not content:  # Couldn't get anything from the wiki
        return

    list_content = _extract_wiki_list(content)
    parsed_items = _parse_wiki_list(list_content)
    the_dict = {key: value for key, value in parsed_items}
    file_io_util.write_json_file(the_dict, cache_path)


def _fetch_wiki_page_raises(dialog_help_url: str) -> str:
    """Get the content of the wiki page for the currently running app.

    Args:
        dialog_help_url: Wiki page with all help urls, e.g. 'https://www.xmswiki.com/wiki/GMS:GMS_10.5_Dialog_Help'
    """
    req = urllib.request.Request(
        url=dialog_help_url, data=None, headers={'User-Agent': f"Guipy {guipy_version} (XMS-like)"}
    )
    with urllib.request.urlopen(req) as f:
        content = f.read().decode('utf-8')
    return content


def _fetch_wiki_page(dialog_help_url: str) -> str:
    """Get the content of the wiki page for the currently running app.

    Args:
        dialog_help_url: Wiki page with all help urls, e.g. 'https://www.xmswiki.com/wiki/GMS:GMS_10.5_Dialog_Help'
    """
    try:
        return _fetch_wiki_page_raises(dialog_help_url)
    except URLError as e:
        print(str(e))
        return ''


def _extract_wiki_list(content: str) -> str:
    """Find the <ul> or <ol> tag containing the dialog help list and extract it.

    Args:
        content: The entire page's content.

    Returns:
        The part of the content between the opening and closing list tags, excluding the open and close tags.
    """
    header_start = content.find('Wiki Page')
    if header_start < 0:
        raise AssertionError('Dialog help page format seems to have changed.')

    ol_start = content.find('<ol>', header_start)
    ul_start = content.find('<ul>', header_start)
    if ol_start < 0 and ul_start < 0:
        raise AssertionError('Help page format may have changed')
    elif ol_start < 0:
        tag_name = 'ul'
    elif ul_start < 0:
        tag_name = 'ol'
    elif ul_start < ol_start:
        tag_name = 'ul'
    else:
        tag_name = 'ol'

    start_tag = f'<{tag_name}>'
    start = content.find(start_tag, header_start) + len(start_tag)  # Skip the start tag
    end = content.find(f'</{tag_name}>', header_start)

    list_content = content[start:end]
    return list_content


def _parse_wiki_list(content: str) -> list[tuple[str, str]]:
    """Parse the <li> tags in the dialog help wiki page.

    Args:
        content: Everything between the opening <ul> or <ol> tag and the matching close one.

    Returns:
        List of tuples of (key, url).
    """
    mapping = []
    items = content.split('<li>')
    items.pop(0)  # We should start with an <li>, which means the first item is always empty
    for item in items:
        start = 9
        end = item.find('"', start)
        url = item[start:end]

        start = item.rfind('|') + 1
        end = item.rfind('</li>')
        key = item[start:end].strip()

        mapping.append((key, url))
    return mapping


def main(args: list[str]) -> None:
    """Runs the script.

    Args:
        args (list[str]): Command line args:
            0: script filepath
            1: cache filepath
            2: dialog_help_url
            3: test_fetch_time (optional)
    """
    if len(args) > 3:
        global _test_fetch_time
        _test_fetch_time = int(args[3])
    _create_url_cache(Path(args[1]), args[2])


if __name__ == '__main__':
    main(sys.argv)  # pragma: no cover
