"""Widget for the font selector button."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Signal, Slot
from PySide2.QtGui import QColor, QFont, QPalette
from PySide2.QtWidgets import QFontDialog, QPushButton

# 3. Aquaveo modules

# 4. Local modules


class FontButton(QPushButton):
    """QPushButton implementation for selecting fonts."""
    font_changed = Signal()
    current_font_changed = Signal(QFont)

    def __init__(self, parent=None):
        """Initializes the widget.

        Args:
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(parent)

        self.color = QColor()
        self.font = QFont("Arial", 8)

        self.clicked.connect(self.font_clicked)
        self.current_font_changed[QFont].connect(self.font_changed)

    @Slot()
    def font_clicked(self):
        """Called when a button that is invokes the font dialog is clicked."""
        ok, new_font = QFontDialog.getFont(self.font, self)

        if ok:
            self.set_font(new_font)

    @Slot(QFont)
    def set_font(self, new_font):
        """Sets the current font. This is called when the user chooses a new font.

        Args:
            new_font (QFont): The new current font.
        """
        self.font = new_font
        self.get_font_text()
        self.current_font_changed.emit(self.font)

    @Slot(QColor)
    def set_color(self, new_color):
        """Sets the current color. This is called when the user chooses a new color.

        Args:
            new_color (QColor): The new current color.
        """
        self.color = new_color
        self.get_font_text()

    def get_font_text(self):
        """Draws text in the current font and color."""
        p = QPalette()
        p.setColor(QPalette.ButtonText, self.color)

        self.setPalette(p)
        self.setFont(self.font)
        self.setText("AaBb")
