"""A Qt GroupBox that can be expanded and collapsed."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2 import QtCore
from PySide2.QtWidgets import QFrame, QGridLayout, QScrollArea, QSizePolicy, QToolButton, QWidget

# 3. Aquaveo modules

# 4. Local modules


class QxExpandableSection(QWidget):
    """A Qt GroupBox that can be expanded and collapsed."""
    def __init__(self, parent=None, title='', animation_duration=300):
        """Initializes the class.

        Adapted from c++ version.
            http://stackoverflow.com/questions/32476006/how-to-make-an-expandable-collapsable-section-widget-in-qt

        Args:
            parent (Something derived from QWidget): The parent window.
            title (str): The string to display next to the arrow.
            animation_duration (int): The duration of the animation.
        """
        super().__init__(parent=parent)
        self.animation_duration = animation_duration
        self.toggle_animation = QtCore.QParallelAnimationGroup()
        self.content_area = QScrollArea()
        self.header_pline = QFrame()
        self.toggle_button = QToolButton()
        self.main_layout = QGridLayout()

        toggle_button = self.toggleButton
        toggle_button.setStyleSheet("QToolButton { border: none; }")
        toggle_button.setToolButtonStyle(QtCore.Qt.ToolButtonTextBesideIcon)
        toggle_button.setArrowType(QtCore.Qt.RightArrow)
        toggle_button.setText(str(title))
        toggle_button.setCheckable(True)
        toggle_button.setChecked(False)

        header_line = self.header_pline
        header_line.setFrameShape(QFrame.HLine)
        header_line.setFrameShadow(QFrame.Sunken)
        header_line.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Maximum)

        self.content_area.setStyleSheet("QScrollArea { background-color: white; border: none; }")
        self.content_area.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
        # start out collapsed
        self.content_area.setMaximumHeight(0)
        self.content_area.setMinimumHeight(0)
        # let the entire widget grow and shrink with its content
        toggle_animation = self.toggle_animation
        toggle_animation.addAnimation(QtCore.QPropertyAnimation(self, b"minimumHeight"))
        toggle_animation.addAnimation(QtCore.QPropertyAnimation(self, b"maximumHeight"))
        toggle_animation.addAnimation(QtCore.QPropertyAnimation(self.content_area, b"maximumHeight"))
        # don't waste space
        main_layout = self.main_layout
        main_layout.setVerticalSpacing(0)
        main_layout.setContentsMargins(0, 0, 0, 0)
        row = 0
        main_layout.addWidget(self.toggleButton, row, 0, 1, 1, QtCore.Qt.AlignLeft)
        main_layout.addWidget(self.header_pline, row, 2, 1, 1)
        row += 1
        main_layout.addWidget(self.content_area, row, 0, 1, 3)
        self.setLayout(self.main_layout)

        # self.toggleButton.clicked.connect(self.start_animation)
        self.toggleButton.toggled.connect(self.start_animation)  # Fires on setChecked so we can start out open.

    def start_animation(self):
        """Slot called when toggle button is clicked. Starts the animation."""
        self.start_animation2(self.toggleButton.isChecked())

    def start_animation2(self, checked):
        """Starts the animation. If checked is True, expands, else contracts.

        Args:
            checked (bool): True to expand, False to contract.

        """
        arrow_type = QtCore.Qt.DownArrow if checked else QtCore.Qt.RightArrow
        direction = QtCore.QAbstractAnimation.Forward if checked else QtCore.QAbstractAnimation.Backward
        self.toggleButton.setArrowType(arrow_type)
        self.toggle_animation.setDirection(direction)
        self.toggle_animation.start()

    def setContentLayout(self, content_layout):  # noqa: N802
        """Sets the layout that will be inside the expandable section.

        Args:
            content_layout (QLayout derivative): The layout.

        """
        # Not sure if this is equivalent to self.content_area.destroy()
        self.content_area.destroy()
        self.content_area.setLayout(content_layout)
        collapsed_height = self.sizeHint().height() - self.content_area.maximumHeight()
        content_height = content_layout.sizeHint().height()
        for i in range(self.toggle_animation.animationCount() - 1):
            spoiler_animation = self.toggle_animation.animationAt(i)
            spoiler_animation.setDuration(self.animation_duration)
            spoiler_animation.setStartValue(collapsed_height)
            spoiler_animation.setEndValue(collapsed_height + content_height)
        content_animation = self.toggle_animation.animationAt(self.toggle_animation.animationCount() - 1)
        content_animation.setDuration(self.animation_duration)
        content_animation.setStartValue(0)
        content_animation.setEndValue(content_height)
