"""HgsExporter class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from contextlib import contextmanager
import logging
from pathlib import Path

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.dmi import XmsEnvironment
from xms.api.tree import tree_util, TreeNode
from xms.components.display import windows_gui
from xms.guipy.dialogs import xms_parent_dlg
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg
from xms.guipy.dialogs.process_feedback_thread import ProcessFeedbackThread
from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists

# 4. Local modules
from xms.hgs.file_io import grok_writer


@contextmanager
def parent_window_context(testing: bool):
    """Context manager to return the parent window and start/stop the timer that raises the previous window."""
    ensure_qapplication_exists()
    win_cont = None
    main_hwnd = None
    if not testing:
        # Parse parent HWNDs and icon path from commandline arguments.
        parent_hwnd, main_hwnd, _icon_path = xms_parent_dlg.parse_parent_window_command_args()
        win_cont = xms_parent_dlg.get_parent_window_container(parent_hwnd)
        # Create the timer that keeps our Python dialog in the foreground of XMS.
        _ = windows_gui.create_and_connect_raise_timer(main_hwnd, win_cont)  # Keep the timer in scope
    try:
        yield win_cont
    finally:
        if win_cont:
            windows_gui.raise_main_xms_window(main_hwnd)  # Bring top-level Python window to foreground


class HgsExporter:
    """Handles the Save Simulation and Export menu commands, the ProcessFeedback dialog, and do_work thread.

    The actual writing to disc is delegated to the GrokWriter class so that this class isn't so bloated.
    """
    def __init__(self, main_file: str, query: Query, out_dir: Path, sim_node) -> None:
        """Initializes the class.

        Args:
            main_file (str): Simulation component main file.
            query (Query): Object for communicating with GMS
            out_dir (Path): Path to the output directory.
            sim_node (TreeNode): Simulation tree node.
        """
        self._main_file: str = main_file
        self._query = query
        self._out_dir = out_dir
        self._sim_node = sim_node
        self._log = logging.getLogger('xms.hgs')

    # def _clear_directory(self) -> None:
    #     """Clears everything from the directory where we are about to write the model native files."""
    #     self._log.info('Clearing the directory...')
    #     filesystem.clear_folder(str(self._out_dir))

    def _write(self) -> str:
        """Writes the simulation."""
        self._log.info('Writing simulation...')
        rv = grok_writer.write(self._main_file, self._query, self._out_dir, self._sim_node)
        return str(rv) if rv is not None else ''

    @staticmethod
    def _create_query() -> Query:  # pragma no cover - can't test this without an XMS running
        """Creates and returns a query.

        Returns:
            See description.
        """
        query = Query(timeout=300000)
        return query

    def do_work(self) -> None:
        """Exports this HydroGeoSphere simulation."""
        # self._clear_directory()  This probably isn't a good idea.
        grok_filename = self._write()
        if grok_filename:
            self._log = logging.getLogger('xms.hgs')
            self._log.info('Simulation export complete.\n')

    @staticmethod
    def _run(
        main_file: str,
        query: Query,
        out_dir: Path,
        sim_node: TreeNode,
        feedback: bool = True,
        testing: bool = False
    ) -> None:
        """Runs the simulation exporter.

        Args:
            main_file (str): Simulation component main file.
            query (xmsapi.dmi.Query): Object for communicating with GMS
            out_dir (str): Path where simulation is to be exported.
            sim_node (TreeNode): Simulation tree node.
            feedback (bool): True if the feedback dialog is to be shown.
            testing (bool): Passing true closes the dialog automatically.
        """
        with parent_window_context(testing) as win_cont:
            HgsExporter._run_feedback_dialog(main_file, query, out_dir, sim_node, feedback, testing, win_cont)

    @staticmethod
    def _run_feedback_dialog(
        main_file: str, query: Query, out_dir: Path, sim_node: TreeNode, feedback: bool, testing: bool,
        win_cont: QWidget
    ) -> None:
        """Runs the process feedback dialog.

        Args:
            main_file (str): Simulation component main file.
            query (xmsapi.dmi.Query): Object for communicating with GMS
            out_dir (str): Path where simulation is to be exported.
            sim_node (TreeNode): Simulation tree node.
            feedback (bool): True if the feedback dialog is to be shown.
            testing (bool): Passing true closes the dialog automatically.
            win_cont (QWidget): The window container.
        """
        exporter = HgsExporter(main_file=main_file, query=query, out_dir=out_dir, sim_node=sim_node)
        worker = ProcessFeedbackThread(do_work=exporter.do_work, parent=win_cont)
        display_text = {
            'title': 'Saving HydroGeoSphere Simulation',
            'working_prompt': f'Saving HydroGeoSphere simulation to \"{out_dir}\".',  # noqa: B028
            'error_prompt': 'Error(s) encountered while saving.',
            'warning_prompt': 'Warning(s) encountered while saving.',
            'success_prompt': f'Successfully exported \"{out_dir}\".',  # noqa: B028
            'note': '',
            # 'auto_load': 'Close this dialog automatically when exporting is finished.',
            'log_format': '%(asctime)s - %(message)s',
            'use_colors': True
        }
        dlg = ProcessFeedbackDlg(display_text=display_text, logger_name='xms.hgs', worker=worker, parent=win_cont)
        dlg.testing = not feedback or testing
        if dlg.exec():
            if dlg.logged_error():
                XmsEnvironment.report_export_error()
        else:
            XmsEnvironment.report_export_aborted()

    @staticmethod
    def run_script(feedback: bool = True, testing: bool = False) -> None:
        """Runs the simulation exporter when called as a script.

        Args:
            feedback (bool): True if the feedback dialog is to be shown. This
            creates a separate worker thread which makes debugging difficult.
            testing (bool): Passing true closes the dialog automatically.
        """
        query = HgsExporter._create_query()
        sim_node = tree_util.find_tree_node_by_uuid(query.project_tree, query.current_item_uuid())
        main_file = query.item_with_uuid(
            item_uuid=sim_node.uuid, model_name="HydroGeoSphere", unique_name="Sim_Manager"
        ).main_file
        out_dir = Path.cwd()
        HgsExporter._run(
            main_file=main_file, query=query, out_dir=out_dir, sim_node=sim_node, feedback=feedback, testing=testing
        )

    @staticmethod
    def run_export(main_file: str, query: Query, out_dir: Path, feedback: bool = True) -> None:
        """Runs the simulation exporter when called from the component Export command.

        Args:
            main_file (str): Simulation component main file.
            query (Query): Object for communicating with GMS
            out_dir (Path): Path where simulation is to be exported.
            feedback (bool): True if the feedback dialog is to be shown.
        """
        sim_node = tree_util.find_tree_node_by_uuid(query.project_tree, query.parent_item_uuid())
        HgsExporter._run(main_file=main_file, query=query, out_dir=out_dir, sim_node=sim_node, feedback=feedback)
