"""SimComponent class."""

__copyright__ = '(C) Copyright Aquaveo 2022'
__license__ = 'All rights reserved'
__all__ = ['SimComponent']

# 1. Standard Python modules
from functools import cached_property
import os
from pathlib import Path
from typing import Optional

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment as XmEnv
from xms.components.bases.component_with_menus_base import MessagesAndRequests
from xms.components.component_builders.coverage_component_builder import CoverageComponentBuilder
from xms.gmi.component_bases.sim_component_base import SimComponentBase
from xms.gmi.data.generic_model import GenericModel, Section
from xms.gmi.gui.section_dialog import SectionDialog
from xms.guipy.dialogs.process_feedback_dlg import run_feedback_dialog

# 4. Local modules
from xms.hydroas.components.coverage_component import CoverageComponent
from xms.hydroas.components.material_component import MaterialComponent
from xms.hydroas.components.template_finder import find_template
from xms.hydroas.data.data import get_version
from xms.hydroas.data.sim_data import SimData
from xms.hydroas.dmi.xms_data import XmsData
from xms.hydroas.feedback.map_materials_runner import MapMaterialsRunner
from xms.hydroas.feedback.map_roughness_runner import MapRoughnessRunner


class SimComponent(SimComponentBase):
    """A hidden Dynamic Model Interface (DMI) component for the HYDRO_AS-2D model simulation."""
    def __init__(self, main_file: Optional[str | Path] = None):
        """
        Initializes the component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.tree_commands.append(('Add BC Coverage', self.add_bc_coverage))
        self.tree_commands.append(('Add Material Coverage', self.add_material_coverage))
        self.default_tree_command = self._internal_open_model_control
        self._create_event_handlers.append(self._check_template)

    @cached_property
    def data(self) -> SimData:
        """The component's data manager."""
        return SimData(self.main_file)

    def _get_global_parameter_section(self) -> Section:
        """Get the global parameter section."""
        return self.data.generic_model.global_parameters

    def _open_model_control(self, dialog_name: str, query: Query, parent: QWidget):
        """
        Run the model control dialog.

        Args:
            dialog_name: A name that can be used as a registry key for loading and saving dialog settings, such as its
                geometry and active selections.
            query: Interprocess communication object.
            parent: Parent widget for the dialog.
        """
        self._query = query
        section = self._get_global_parameter_section()
        values = self.data.global_values
        section.restore_values(values)

        dlg = SectionDialog(
            parent=parent,
            section=section,
            dlg_name=dialog_name,
            window_title='Model control',
            **self._section_dialog_keyword_args,
        )
        if dlg.exec():
            values = dlg.section.extract_values(include_default=False)
            self.data.global_values = values
            self.data.commit()

    def item_linked(self, query: Query, linked_uuid: str, unique_name: str, lock_state: bool, parent: QWidget) -> None:
        """
        Handle when a new item was linked to the simulation.

        Args:
            query: Interprocess communication object.
            linked_uuid: UUID of the item that was just linked.
            unique_name: The unique-name of the item being linked, assuming the item's XML was designed the way this
                component expects.

                The sim component's `<component>` tag should have one `<takes>` tag inside for each type of thing that
                can be linked to the simulation. Each `<takes>` tag should have a `<declare_parameter>` tag inside it.
                The `unique_name` passed to this method will be the value of the `<declare_parameter>` tag for the item
                being linked. The value of this tag should probably be the item's unique-name (typically its class name)
                since it's basically the same thing.
            lock_state: Whether the item is currently locked for editing. Currently only makes sense in GMS.
            parent: Parent widget.
        """
        data = XmsData(query)
        if unique_name == 'UGrid2d':
            link_ugrid(data, linked_uuid, parent)
        elif unique_name == 'RoughnessComponent':
            link_roughness(data, self.data.generic_model, linked_uuid, parent)
        elif unique_name == data.bc_component_unique_name:
            messages = link_bc(data, self.data.generic_model, linked_uuid)
            self.messages.extend(messages)
        elif unique_name == data.material_component_unique_name:
            messages = link_materials(data, self.data.generic_model, linked_uuid, parent)
            self.messages.extend(messages)
        else:
            data.unlink(linked_uuid)
            self.messages.append(('ERROR', 'Unrecognized item type'))

    def _check_template(self, _query: Query):
        """Check if the component needs its template initialized and, if necessary, schedule the initialization."""
        if self.data.generic_model.is_default():
            action = self._make_request(self._get_template)
            self.requests.append(action)

    def _get_template(self, query: Query, _params: list[dict], parent: QWidget) -> MessagesAndRequests:
        """
        Initialize the component's template.

        Args:
            query: Interprocess communication object.
            _params: Ignored.
            parent: Parent widget for dialogs.

        Returns:
            Messages and requests.
        """
        return find_template(self.data, query, parent)

    def add_bc_coverage(self, query: Query, _params: list[dict], _parent: QWidget):
        """
        Add a boundary condition coverage based on the simulation.

        Args:
            query: Interprocess communication object.
            _params: Ignored.
            _parent:  Ignored.

        Returns:
            Empty messages and requests.
        """
        version = get_version(self.data.generic_model, self.data.model_values)
        name = f'New Boundary Conditions [{version}]'

        builder = CoverageComponentBuilder(CoverageComponent, name, None)
        builder.data.generic_model = self.data.generic_model
        coverage, component, keywords = builder.build()

        data = XmsData(query)
        data.add_coverage(coverage, component, keywords=keywords)

        return [], []

    def add_material_coverage(self, query: Query, _params: list[dict], _parent: QWidget) -> MessagesAndRequests:
        """
        Add a material coverage based on the simulation.

        Args:
            query: Interprocess communication object.
            _params: Ignored.
            _parent:  Ignored.

        Returns:
            Empty messages and requests.
        """
        version = get_version(self.data.generic_model, self.data.model_values)
        name = f'New Materials [{version}]'

        builder = CoverageComponentBuilder(MaterialComponent, name, None)
        builder.data.generic_model = self.data.generic_model

        coverage, component, keywords = builder.build()

        data = XmsData(query)
        data.add_coverage(coverage, component, keywords=keywords)

        return [], []


def link_ugrid(data: XmsData, linked_uuid: str, parent: QWidget):
    for linked_ugrid_uuid in data.linked_ugrid_uuids:
        if linked_ugrid_uuid != linked_uuid:
            data.unlink(linked_ugrid_uuid)

    data.unlink_materials()


def link_roughness(data: XmsData, model: GenericModel, linked_uuid: str, parent: QWidget):
    runner = MapRoughnessRunner(data, model, linked_uuid)
    run_feedback_dialog(runner, parent=parent)


def _compare_models(data: XmsData, sim_model: GenericModel, linked_uuid: str,
                    unique_name: str) -> list[tuple[str, str]]:
    coverage_data = data.get_component_data(linked_uuid, unique_name)
    linked_model = coverage_data.generic_model
    if sim_model.compatible(linked_model):
        return []

    directory = Path(XmEnv.xms_environ_temp_directory()) / 'gmi-rejected-link-info'
    os.makedirs(directory, exist_ok=True)
    with open(directory / 'gmi_simulation_template.txt', 'w') as f:
        f.write(sim_model.to_pretty_string())
    with open(directory / 'gmi_coverage_template.txt', 'w') as f:
        f.write(linked_model.to_pretty_string())

    data.unlink(linked_uuid)
    return [('ERROR', 'Linked item was not compatible with simulation.')]


def link_bc(data: XmsData, sim_model: GenericModel, linked_uuid: str) -> list[tuple[str, str]]:
    return _compare_models(data, sim_model, linked_uuid, data.bc_component_unique_name)


def link_materials(data: XmsData, sim_model: GenericModel, linked_uuid: str, parent: QWidget) -> list[tuple[str, str]]:
    error_message = _compare_models(data, sim_model, linked_uuid, data.material_component_unique_name)
    if error_message:
        return error_message

    runner = MapMaterialsRunner(data)
    run_feedback_dialog(runner, parent=parent)
    return []
