"""Various error messages that can be generated while reading a file."""

# 1. Standard Python modules
from typing import NoReturn

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class GmiError(Exception):
    """Represents a parsing error in a .2dm file."""
    pass


class Messages:
    """Error messages for reading/writing a 2DM file."""

    # Read errors
    ambiguous_correlation = 'Name defined by multiple GP cards'
    ambiguous_parent = 'Multiple option parameters found with name'
    bad_entity_short_name = 'Entity ID must be N, S, or E'
    bad_entity_group = 'Unknown group'
    bad_dependency_type = 'Invalid dependency type'
    bad_mode = 'Invalid mode'
    bad_option_value = 'Value not in allowed values for option parameter'
    bad_international = '"International" field only allowed on feet'
    defined_disabled_material = 'Disabled material is implicitly defined'
    duplicate_card = 'Parameter already set on line {0}'
    duplicate_id = 'ID already defined on line {0}'
    extra_field = 'Extra field'
    extra_negative_node = 'More than one negative node ID'
    missing_field = 'Missing field'
    missing_nodestring_id = 'Missing nodestring ID'
    missing_negative_node = 'Missing negative node ID'
    missing_parent = 'No option parameter found with given name'
    mistyped_field = 'Value is not {0}'
    node_after_end = 'Node ID after nodestring ID'
    not_float_or_curve = 'Value must be either "FLOAT" or "CURVE"'
    not_integer_or_string = 'Value is not integer or quoted string'
    not_international = 'Field must be "International" or omitted'
    option_mismatch = 'Option value and enabled flag must come in pairs'
    option_bad_default = 'Parameter default value not in allowed options'
    range_error = 'Value out of range'
    range_error_defined = 'Value outside of range defined for parameter'
    range_inverted = 'Minimum is greater than maximum'
    short_nodestring = 'Nodestring must have at least two nodes'
    target_not_option = 'Target parameter is not of type option'
    undefined_group_name = 'Undefined group name'
    undefined_id = 'Undefined ID'
    undefined_target = 'Target does not exist'
    unknown_card = 'Card "{0}" not allowed in this context'
    unclosed_string = "Unclosed string"
    unsupported_curve_version = "Unsupported curve version"
    wrong_parameter_count = "Wrong number of parameters"

    # Write errors
    degenerate_arc = (
        "\n"  # This is a long message and half the window is taken up by a time stamp, so start on a new line.
        "A degenerate arc was found near location {0}.\n"
        "This might mean the arc was too short to reach multiple nodes, or that it\n"
        "connected disjoint sections of the domain and its feature type was incompatible\n"
        "with disjoint domains. The identified location is the first node on the arc."
    )
    duplicate_point = "Duplicate point detected near domain node {0}"
    duplicate_arc = "Duplicate arc ID: {0}"
    features_without_domain = "Exporting coverages requires a mesh in the simulation."
    materials_without_domain = 'Exporting mapped materials with cell data requires a mesh in the simulation.'


def error(line_number: int, *args) -> NoReturn:
    """
    Raise GMI error with a message.

    Args:
        line_number: Line the error occurred on
        *args: An optional field number, followed by a message, followed by anything to insert into the message.
    """
    if isinstance(args[0], int):
        field_number = args[0]
        message = args[1]
        rest = args[2:]
    else:
        field_number = 0
        message = args[0]
        rest = args[1:]

    message = message.format(*rest)

    if field_number > 0:
        message = f'Error on line {line_number}, field {field_number}: {message}.'
    else:
        message = f'Error on line {line_number}: {message}.'

    raise GmiError(message)


def write_error(message: str, *args) -> NoReturn:
    """
    Raise an error when writing a .2dm file.

    Args:
        message: The message to write.
        args: Items to format into the message.
    """
    message = message.format(*args)
    raise GmiError(message)
