"""CbcZoneBudgetRunner class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass
import os
from subprocess import PIPE, Popen, STDOUT

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util
from xms.components.bases.component_base import ActionRv
from xms.guipy.dialogs import process_feedback_dlg
from xms.guipy.dialogs.feedback_thread import FeedbackThread

# 4. Local modules
from xms.mf6.components import dmi_util
from xms.mf6.misc import log_util


@dataclass
class ZbInput:
    """All input needed to run ZONEBUDGET, so we don't have to pass all these args separately every time."""
    cbc_filename: str  # file name for the modflow budget file.
    zb_exe: str  # Path to the zone budget executable file.
    zb_name_file: str  # Path to the zone budget name file.
    sim_uuid: str  # uuid of the simulation
    model_uuid: str  # uuid of the model
    comp_dir: str  # directory for components


def run(zb_in: ZbInput, query: Query, win_cont: QWidget) -> ActionRv:
    """Runs ZONEBUDGET and adds the result files to the existing solution.

    Args:
        zb_in: All input needed to run ZONEBUDGET.
        query: a Query object to communicate with GMS.
        win_cont: The window container.
    """
    thread = ZonebudgetRunnerFeedbackThread(query, zb_in)
    process_feedback_dlg.run_feedback_dialog(thread, win_cont)
    return [], []


class ZonebudgetRunnerFeedbackThread(FeedbackThread):
    """Thread for reading running ZONEBUDGET."""
    def __init__(self, query: Query, zb_in: ZbInput):
        """Initializes the class.

        Args:
            zb_in: All input needed to run ZONEBUDGET.
            query: a Query object to communicate with GMS.
        """
        super().__init__(query)
        self._query = query
        self._zb_in: ZbInput = zb_in
        self.display_text |= {
            'title': 'MODFLOW 6 Zone Budget',
            'working_prompt': 'Running zone budget...',
            'error_prompt': 'Error(s) encountered.',
            'warning_prompt': 'Warning(s) encountered.',
            'success_prompt': 'No warnings or errors encountered.',
            'note': f'Using ZONEBUDGET 6 executable: {self._zb_in.zb_exe}',
            'auto_load': ''  # This makes it stay open
        }

    def _run(self):
        """Does the work."""
        runner = ZoneBudgetRunner(self._zb_in, self._query)
        runner.run()


class ZoneBudgetRunner:
    """Runs ZONEBUDGET."""
    def __init__(self, zb_in: ZbInput, query: Query):
        """Initializes the class.

        Args:
            zb_in: All input needed to run ZONEBUDGET.
            query: a Query object to communicate with GMS.
        """
        self._zb_in: ZbInput = zb_in
        self._query: Query = query

        self._zb_nam_file_dir = os.path.dirname(self._zb_in.zb_name_file)
        self._folder_path: str = ''
        self._normal_termination = False
        self._node = None
        self._out_name = []
        self._log = log_util.get_logger()

    def run(self):
        """Runs zone budget."""
        self._log.info(f'Process ID: {os.getpid()}')

        base_name = os.path.splitext(os.path.basename(self._zb_in.zb_name_file))[0]
        self._out_name.append(f'{base_name}.lst')
        self._out_name.append(f'{base_name}.csv')

        self._remove_previous_solution()
        self._run_zone_budget_exe()
        self._add_files_to_solution()

    def _remove_previous_solution(self):
        """Remove the previous solution from the project explorer."""
        cbc_file_node = tree_util.find_tree_node_by_uuid(self._query.project_tree, self._query.current_item_uuid())
        solution_node = cbc_file_node.parent.parent
        zb_solution_node = tree_util.child_from_name(solution_node, os.path.basename(self._zb_nam_file_dir))
        if not zb_solution_node:
            return

        for child in zb_solution_node.children:
            self._query.delete_item(child.uuid)
        self._query.delete_item(zb_solution_node.uuid)

    def _run_zone_budget_exe(self):
        """Runs the zone budget executable."""
        os.chdir(self._zb_nam_file_dir)
        cmd = f'"{self._zb_in.zb_exe}" "{os.path.basename(self._zb_in.zb_name_file)}"'
        process = Popen(cmd, stdout=PIPE, stderr=STDOUT, shell=True)
        while True:
            line = process.stdout.readline()
            if not line:
                break
            line = line.decode('utf-8').rstrip()
            if line.find('Normal Termination') > -1:
                self._normal_termination = True
            self._log.info(line)

        if not self._normal_termination:
            self._log.error(' ')  # String must not be empty
        process.stdout.close()
        process.wait()

    def _add_files_to_solution(self):
        """Adds the files that ZONEBUDGET created to the solution folder."""
        if not self._normal_termination:
            return
        for out_name in self._out_name:
            out_name = os.path.join(self._zb_nam_file_dir, out_name)
            txt_comp = dmi_util.build_solution_component(
                out_name, self._zb_in.sim_uuid, self._zb_in.model_uuid, self._zb_in.comp_dir, 'TXT_SOL'
            )
            # Add the component to the Context
            self._query.add_component(txt_comp, folder_path=os.path.basename(self._zb_nam_file_dir))
