"""CtpData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.list_package_data import ListPackageData
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import gui_util
from xms.mf6.gui.options_defs import Checkbox, CheckboxButton, CheckboxField


class CtpData(ListPackageData):
    """Data class to hold the info from a CTP package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'CTP6'

    def get_column_tool_tips(self, block: str) -> dict[int, str]:
        """Returns a dict with column index and tool tip.

        Args:
            block (str): Name of the block.
        """
        names, _types, _defaults = self.get_column_info(block)
        return {names.index('TEMP'): 'Temperature at the boundary'}

    def package_column_info(self, block=''):
        """Returns the column info just for the columns unique to this package.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        columns = {'TEMP': (object, 0.0)}
        return gui_util.column_info_tuple_from_dict(columns)

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Constant Temperature (CTP) Package'

    def map_info(self, feature_type):
        """Returns info needed for Map from Coverage.

        Args:
            feature_type (str): 'points', 'arcs', or 'polygons'

        Returns:
            (dict): Dict describing how to get the MODFLOW variable from the shapefile or att table fields.
        """
        return {'Name': None, 'TEMP': None}

    def map_import_info(self, feature_type):
        """Returns info needed for mapping shapefile or transient data file to package data.

        Args:
            feature_type (str): 'points', 'arcs', or 'polygons'

        Returns:
            (dict): See description
        """
        info = {'TEMP': None}
        self._add_aux_and_boundname_info(info)
        return info

    # @overrides
    def _setup_options(self):
        """Returns the definition of all the available options.

        Returns:
            (OptionsBlock): See description.
        """
        return OptionsBlock(
            [
                CheckboxButton(
                    'AUXILIARY',
                    button_text='Auxiliary Variables...',
                    check_box_method='on_chk_auxiliary',
                    button_method='on_btn_auxiliary'
                ),
                CheckboxField(
                    'AUXMULTNAME', brief='Auxiliary variable used as multiplier of temperature value', type_='str'
                ),
                Checkbox('BOUNDNAMES', brief='Allow boundary names', check_box_method='on_chk_boundnames'),
                Checkbox('PRINT_INPUT', brief='Print input to listing file'),
                Checkbox('PRINT_FLOWS', brief='Print flows to listing file'),
                Checkbox('SAVE_FLOWS', brief='Save flows to budget file'),
                CheckboxButton(
                    'TS6 FILEIN', brief='Time-series files', button_text='Files...', button_method='on_btn_ts6_filein'
                ),
                CheckboxButton(
                    'OBS6 FILEIN',
                    brief='Observation files',
                    button_text='Files...',
                    button_method='on_btn_obs6_filein'
                ),
            ]
        )

    # @overrides
    def append_period_data(self, period, rows):
        """Appends the data to the exiting period info.

        Args:
            period (int): Stress period.
            rows (list): List of lists consisting of rows of data.
        """
        new_rows = self._remove_duplicate_cellids(rows)
        super().append_period_data(period, new_rows)

    def _remove_duplicate_cellids(self, rows):
        """Removes rows where the cellid is a duplicate and returns the corrected rows.

        CHD6 package cannot have duplicate cellids.

        Args:
            rows (list): List of lists consisting of rows of data.

        Returns:
            rows (list): List of lists consisting of rows of data.
        """
        new_rows = []
        cellid_set = set()
        cellid_column_count = self.grid_info().cell_id_column_count()
        for row in rows:
            cellid = tuple(row[:cellid_column_count])
            if cellid not in cellid_set:
                cellid_set.add(cellid)
                new_rows.append(row)
        return new_rows
