"""LktData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data import data_util
from xms.mf6.data.advanced_transport_data_base import AdvancedTransportDataBase
from xms.mf6.gui import gui_util, units_util


class LktData(AdvancedTransportDataBase):
    """Data class to hold the info from an LKT6 package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'LKT6'

    # @overrides
    def get_column_delegate_info(self, block):
        """Returns a list of tuples of [0] column index and [1] list of strings.

        Returns:
            (tuple): tuple containing:
                - index (int): Column index.
                - strings (list of str): List of strings allowed in the column.
        """
        delegate_info = None
        names, _, _ = self.get_column_info(block)
        if not block or block == 'PERIODS':
            delegate_info = [
                (
                    names.index('LAKSETTING'), [
                        'STATUS', 'CONCENTRATION', 'RAINFALL', 'EVAPORATION', 'RUNOFF', 'AUXILIARY'
                    ]
                )
            ]

        return delegate_info

    # @overrides
    def get_column_info(self, block, use_aux=True):
        """Returns column names, types, and defaults.

        The columns depend on the DIS package in use and the AUX variables.
        The package specific and AUX columns are type object because they
        might contain time series strings.

        Args:
            block (str): Name of the list block.
            use_aux (bool): True to include AUXILIARY variables.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        if block.upper() == 'PACKAGEDATA':
            columns = {
                'LAKENO': (np.int32, 1),
                'STRT': (np.float64, 0.0),
            }
            self.add_aux_columns_to_dict(columns, use_aux=use_aux)
            data_util.add_boundname_columns_to_dict(self.options_block, columns)

        else:  # This would be the stress periods
            return self.package_column_info()

        names, types, defaults = gui_util.column_info_tuple_from_dict(columns)
        return names, types, defaults

    def get_column_tool_tips(self, block: str) -> dict[int, str]:
        """Returns a dict with column index and tool tip.

        Args:
            block (str): Name of the block.
        """
        names, _types, _defaults = self.get_column_info(block)
        if block.upper() == 'PACKAGEDATA':
            concentration_units = units_util.string_from_units(self, units_util.UNITS_CONCENTRATION)
            return {
                names.index('LAKENO'): 'Lake number',
                names.index('STRT'): f'Starting concentration for the lake {concentration_units}'
            }
        else:
            return {
                names.index('LAKENO'): 'Lake number',
                names.index('LAKSETTING'): 'Keyword to start LAK setting line'
            }

    def package_column_info(self, block=''):
        """Returns the column info just for the columns unique to this package.

        You should override this method.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        columns = {
            'LAKENO': (np.int32, 1),
            'LAKSETTING': (object, ''),
            'VALUE1': (object, ''),
            'VALUE2': (object, ''),
        }

        names, types, defaults = gui_util.column_info_tuple_from_dict(columns)
        return names, types, defaults

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Lake Transport (LKT) Package'

    def stress_id_columns(self):
        """Returns the column name where the id exists that can be used to help identify this stress across periods.

        Typically is 'CELLIDX' which is added by GMS but is 'RNO' for SFR.

        Returns:
            See description.
        """
        return ['LAKENO']

    # def map_info(self, feature_type):
    #     """Returns info needed for Map from Coverage.
    #
    #     Args:
    #         feature_type (str): 'points', 'arcs', or 'polygons'
    #
    #     Returns:
    #         (dict): Dict describing how to get the MODFLOW variable from the shapefile or att table fields.
    #     """
    #     if feature_type == 'arcs':
    #         return {'FLOW': None, 'RUNOFF': None, 'ETSW': None, 'PPTSW': None, 'ROUGHCH': None, 'ROUGHBK': None,
    #                 'CDPTH': None, 'FDPTH': None, 'AWDTH': None, 'BWDTH': None, 'HCOND1': None, 'THICKM1': None,
    #                 'ELEVUP': None, 'WIDTH1': None, 'DEPTH1': None, 'HCOND2': None, 'THICKM2': None,
    #                 'ELEVDN': None, 'WIDTH2': None, 'DEPTH2': None, }
    #     else:
    #         return {}
