"""AddStressesDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from enum import IntEnum

# 2. Third party modules
import numpy as np
from PySide2.QtWidgets import QComboBox, QLineEdit, QSpinBox

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.mf6.gui import gui_util, table_gui
from xms.mf6.gui.add_stresses_dialog_ui import Ui_AddStressesDialog


class WhichPeriods(IntEnum):
    """Enumeration for which periods."""
    ALL_DEFINED = 0
    ONLY_THIS = 1


def all_cell_id_column_names() -> set[str]:
    """Return the set of all names that can be used as a stress id - a cell id."""
    return {
        'LAY', 'ROW', 'COL', 'CELL2D', 'CELLID', 'LAY1', 'ROW1', 'COL1', 'CELL2D1', 'CELLID1', 'LAY2', 'ROW2', 'COL2',
        'CELL2D2', 'CELLID2'
    }


class AddStressesData:
    """Data from the AddStressesDialog."""
    def __init__(self, row_data, rows_to_add, which_periods):
        """Initializes the class.

        Args:
            row_data: Data for a row.
            rows_to_add: The rows to add.
            which_periods: The periods to add to.
        """
        self.row_data = row_data
        self.rows_to_add = rows_to_add
        self.which_periods = which_periods


class AddStressesDialog(XmsDlg):
    """A dialog that lets the user add stresses to a package."""
    def __init__(self, data, selected_count, which_periods, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            data (Something derived from BaseFileData): The data class.
            selected_count (int): Number of grid cells selected. If non-zero, 'Number of rows to add' and
             CELLID rows are disabled.
            which_periods (WhichPeriods): 0=All defined periods, 1=Only this period
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(parent, 'xms.mf6.gui.add_stresses_dialog')

        self._data = data
        self.help_getter = gui_util.help_getter(gui_util.help_id_from_key('AddStressesDialog'))

        self.ui = Ui_AddStressesDialog()
        self.ui.setupUi(self)
        self.ui.btn_box.helpRequested.connect(self.help_requested)

        # Setup table
        column_names, column_types, defaults = self._data.get_column_info('PERIODS')
        # self.ui.table_widget.setRowCount(len(column_names))
        column_count = 2
        self.ui.table_widget.setColumnCount(column_count)
        self.ui.table_widget.setHorizontalHeaderLabels(['Column', 'Value'])
        self.ui.table_widget.verticalHeader().hide()
        delegate_info = self._data.get_column_delegate_info('PERIODS')
        for i, name in enumerate(column_names):
            if column_types[name] == np.float64:
                table_gui.add_double_edit_row(name, defaults[name], self.ui.table_widget, False)
            elif column_types[name] == np.int64 or column_types[name] == np.int32:
                if selected_count and name in all_cell_id_column_names():
                    widget = table_gui.add_int_spin_row(
                        label=name,
                        default='from selected cells',
                        minimum=0,
                        maximum=1,
                        table_widget=self.ui.table_widget,
                        locked=False
                    )
                    widget.setEnabled(False)
                else:
                    table_gui.add_int_spin_row(
                        label=name,
                        default=defaults[name],
                        minimum=1,
                        maximum=1e7,
                        table_widget=self.ui.table_widget,
                        locked=False
                    )
            else:
                found_delegate = False
                if delegate_info:
                    for info in delegate_info:
                        if info[0] == i:
                            table_gui.add_combo_box_row(
                                label=name,
                                options_list=info[1],
                                default=0,
                                table_widget=self.ui.table_widget,
                                locked=False
                            )
                            found_delegate = True
                            break
                if not found_delegate:
                    table_gui.add_text_edit_row(name, defaults[name], self.ui.table_widget, False)

        self.ui.table_widget.horizontalHeader().setStretchLastSection(True)
        self.ui.table_widget.resizeColumnToContents(0)
        self.ui.table_widget.resizeColumnToContents(1)
        self.ui.table_widget.resizeRowsToContents()
        table_gui.disable_column(self.ui.table_widget, 0)

        # Rows to add
        self.ui.spn_rows_to_add.setRange(1, 1e7)
        default_number_of_rows_to_add = 1
        rows_to_add = selected_count if selected_count else default_number_of_rows_to_add
        if self._data.ftype == 'HFB6':
            rows_to_add = 1
        self.ui.spn_rows_to_add.setValue(rows_to_add)
        self.ui.spn_rows_to_add.setEnabled(selected_count == 0)
        if selected_count:
            if self._data.ftype == 'HFB6':
                self.ui.txt_number.setText('Number of rows to add:')
            else:
                self.ui.txt_number.setText('Number of rows to add (selected cells):')

        # Which periods
        self.ui.btngrp.setId(self.ui.rbt_all_defined, 0)
        self.ui.btngrp.setId(self.ui.rbt_only_this, 1)
        self.ui.btngrp.button(which_periods).setChecked(True)

    def get_data(self) -> AddStressesData:
        """Returns a tuple: the table data in a list, rows_to_add (int), all_checked (bool).

        Returns:
            See description.
        """
        row_data = {}
        for row in range(self.ui.table_widget.rowCount()):
            widget0 = self.ui.table_widget.item(row, 0)
            widget1 = self.ui.table_widget.cellWidget(row, 1)
            if type(widget1) is QLineEdit:
                row_data[widget0.text()] = widget1.text()
            elif type(widget1) is QSpinBox:
                row_data[widget0.text()] = str(widget1.value())
            elif type(widget1) is QComboBox:
                row_data[widget0.text()] = str(widget1.currentText())
        rv = AddStressesData(
            row_data=row_data, rows_to_add=self.ui.spn_rows_to_add.value(), which_periods=self.ui.btngrp.checkedId()
        )
        return rv
