"""DateTimeDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import (QDateTime, Qt)
from PySide2.QtWidgets import (QCalendarWidget, QDialog, QDialogButtonBox, QTimeEdit, QVBoxLayout)

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules


class DateTimeDialog(XmsDlg):
    """Dialog to get a date/time from the user."""
    def __init__(self, date=None, time=None, parent=None):
        """Initializes the dialog.

        Args:
            date (QDate): The initial date.
            time (QTime): The initial time.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(parent, 'xms.mf6.gui.date_time_dialog')

        self.setWindowTitle("Date/Time")
        vlayout = QVBoxLayout(self)
        self.calendar = QCalendarWidget(self)
        vlayout.addWidget(self.calendar)
        self.time_edit = QTimeEdit(self)
        vlayout.addWidget(self.time_edit)
        self.btn_box = QDialogButtonBox(self)  # We use the variable for testing
        self.btn_box.setStandardButtons(QDialogButtonBox.Cancel | QDialogButtonBox.Ok)
        self.btn_box.accepted.connect(self.accept)
        self.btn_box.rejected.connect(self.reject)
        vlayout.addWidget(self.btn_box)

        if date:
            self.calendar.setSelectedDate(date)
        if time:
            self.time_edit.setTime(time)

    def get_date_time_string(self):
        """Returns the date/time as a string.

        Returns:
            (str): The date/time as a string formatted as Qt.ISODate.
        """
        date = self.calendar.selectedDate()
        time = self.time_edit.time()
        date_time = QDateTime(date, time)
        date_time_string = date_time.toString(Qt.ISODate)
        return date_time_string


def run(date, time, parent=None):
    """Runs the :class:`~xms.mf6.get_int_dialog.GetIntDialog`.

    Args:
        date (QDate): The initial date.
        time (QTime): The initial time.
        parent (Something derived from QWidget): The parent window.

    Returns:
        (tuple): tuple containing:
            - (bool): True or False (for OK or Cancel).
            - (str): the date/time string.
    """
    dialog = DateTimeDialog(date, time, parent)
    if dialog.exec() == QDialog.Accepted:
        return True, dialog.get_date_time_string()
    else:
        return False, ''
