"""GetPorosityDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QButtonGroup, QDialog

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import ComponentItem
from xms.api.tree import tree_util, TreeNode
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.mf6.gui import gui_util
from xms.mf6.gui.get_porosity_dialog_ui import Ui_dlg_get_porosity
from xms.mf6.misc import util


class GetPorosityDialog(XmsDlg):
    """A simple dialog to get an integer value from the user."""
    def __init__(
        self, default_choice: str, constant: float, mst_uuid: str, dataset_uuid: str, cell_count: int,
        project_tree: TreeNode, parent
    ):
        """Initializes the class, sets up the ui.

        Args:
            default_choice (str): Default
            constant (float): The constant value to display.
            mst_uuid (str): MST package uuid.
            dataset_uuid (str): UUID of a dataset to use.
            cell_count (int): Number of UGrid cells.
            project_tree (TreeNode): The project tree.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(parent, 'xms.mf6.gui.get_porosity_dialog')
        self.ui = Ui_dlg_get_porosity()
        self.ui.setupUi(self)

        self._default_choice = default_choice
        self._mst_uuid = mst_uuid
        self._dataset_uuid = dataset_uuid
        self._cell_count = cell_count
        self._project_tree = project_tree
        self._mst_exists = False

        # Initialization
        self._hide_mst_radio_button_if_necessary()
        self._group_radio_buttons()
        self._set_default_choice()
        self._init_fields(constant)

        # Signals
        self.ui.rbt_constant.clicked.connect(self._do_enabling)
        self.ui.rbt_mst_package.clicked.connect(self._do_enabling)
        self.ui.rbt_dataset.clicked.connect(self._do_enabling)
        self.ui.btn_mst_package.clicked.connect(self._on_btn_mst_package)
        self.ui.btn_dataset.clicked.connect(self._on_btn_dataset)

    def _init_fields(self, constant: float) -> None:
        """Initializes all the edit fields (and the spin control)."""
        self.ui.spn_constant.setValue(constant)
        if self._mst_uuid:
            node = tree_util.find_tree_node_by_uuid(self._project_tree, self._mst_uuid)
            if node:
                path = util.tree_node_path(node)
                self.ui.edt_mst_package.setText(path)
        if self._dataset_uuid:
            node = tree_util.find_tree_node_by_uuid(self._project_tree, self._dataset_uuid)
            if node:
                path = util.tree_node_path(node)
                self.ui.edt_dataset.setText(path)

    def _do_enabling(self) -> None:
        """Enable/disable stuff."""
        self.ui.spn_constant.setEnabled(self.ui.rbt_constant.isChecked())
        self.ui.edt_mst_package.setEnabled(self.ui.rbt_mst_package.isChecked())
        self.ui.btn_mst_package.setEnabled(self.ui.rbt_mst_package.isChecked())
        self.ui.edt_dataset.setEnabled(self.ui.rbt_dataset.isChecked())
        self.ui.btn_dataset.setEnabled(self.ui.rbt_dataset.isChecked())

    def _group_radio_buttons(self) -> None:
        """Put radio buttons in a group."""
        btn_group = QButtonGroup(self)
        btn_group.addButton(self.ui.rbt_constant)
        btn_group.addButton(self.ui.rbt_mst_package)
        btn_group.addButton(self.ui.rbt_dataset)

    def _set_default_choice(self) -> None:
        """Sets the initial radio button."""
        self.ui.rbt_constant.setChecked(True)
        if self._mst_exists and self._default_choice == 'mst':
            self.ui.rbt_mst_package.setChecked(True)
        elif self._default_choice == 'dataset':
            self.ui.rbt_dataset.setChecked(True)
        self._do_enabling()

    def _hide_mst_radio_button_if_necessary(self) -> None:
        """Hide the MST radio button if there is no MST package."""
        mst_nodes = tree_util.descendants_of_type(self._project_tree, unique_name='MST6')
        self._mst_exists = bool(mst_nodes)
        if not self._mst_exists:
            self.ui.wid_mst_package.setVisible(False)
            self.ui.rbt_constant.setChecked(True)

    def _selectable_callback_mst(self, node) -> bool:
        """Used by TreeItemSelectorDlg to determine what tree nodes are selectable.

        Args:
            node (TreeNode): A tree node.

        Returns:
            (bool): If the node is selectable.
        """
        return node.unique_name == 'MST6'

    def _selectable_callback_dataset(self, node) -> bool:
        """Used by TreeItemSelectorDlg to determine what tree nodes are selectable.

        Args:
            node (TreeNode): A tree node.

        Returns:
            (bool): If the node is selectable.
        """
        return node.num_vals == self._cell_count

    def _filter_condition_mst(self, node) -> bool:
        """Condition used by util.filter_tree.

        Args:
            node (TreeNode): A node.

        Returns:
            (bool): True if the node should be kept.
        """
        return type(node.data) is ComponentItem and node.unique_name == 'MST6'

    def _on_btn_mst_package(self) -> None:
        trimmed_tree = util.filter_tree(start_node=self._project_tree, condition=self._filter_condition_mst)
        if not trimmed_tree:
            return None

        dialog = TreeItemSelectorDlg(
            title='Select MST Package',
            target_type=ComponentItem,
            pe_tree=trimmed_tree,
            override_icon=gui_util.get_icon_path,
            previous_selection=self._mst_uuid,
            parent=self,
            selectable_callback=self._selectable_callback_mst
        )
        if dialog.exec() == QDialog.Accepted:
            self._mst_uuid = dialog.get_selected_item_uuid()
            node = tree_util.find_tree_node_by_uuid(trimmed_tree, self._mst_uuid)
            self.ui.edt_mst_package.setText(util.tree_node_path(node))

    def _on_btn_dataset(self) -> None:
        dataset_uuid = gui_util.select_dataset_dialog(self, self._project_tree, self._dataset_uuid, self._cell_count)
        if dataset_uuid:
            self._dataset_uuid = dataset_uuid
            node = tree_util.find_tree_node_by_uuid(self._project_tree, self._dataset_uuid)
            self.ui.edt_dataset.setText(util.tree_node_path(node))

    def get_choice(self) -> tuple[str, float | str]:
        """Returns what the user selected."""
        if self.ui.rbt_constant.isChecked():
            return 'constant', self.ui.spn_constant.value()
        elif self.ui.rbt_mst_package.isChecked():
            return 'mst', self._mst_uuid
        elif self.ui.rbt_dataset.isChecked():
            return 'dataset', self._dataset_uuid
