"""GUI utility functions."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import csv
import os
from pathlib import Path
import re
from typing import Any

# 2. Third party modules
import numpy as np
import pandas as pd
from PySide2.QtCore import QSignalBlocker, Qt
from PySide2.QtGui import QPalette
from PySide2.QtWidgets import (QDialog, QFileDialog, QHBoxLayout, QLabel, QSizePolicy, QSpinBox, QTabWidget, QWidget)

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import CoverageItem, DatasetItem
from xms.api.tree import TreeNode
from xms.guipy import settings
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.help_getter import HelpGetter
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.widgets import widget_builder
from xms.guipy.widgets.qx_table_view import QxTableView
from xms.testing.type_aliases import Pathlike

# 4. Local modules
from xms.mf6.data.grid_info import DisEnum, GridInfo
from xms.mf6.file_io import io_util
from xms.mf6.gui.qx_spin_box_style import QxSpinBoxStyle
from xms.mf6.gui.treeitem_selector_with_check_box_ui import Ui_dlg_treeitem_selector_with_check_box
from xms.mf6.gui.treeitem_selector_with_combo_box_ui import Ui_dlg_treeitem_selector_with_combo_box
from xms.mf6.misc import util

# Module variables

# Dialog help ids
_help_ids_dict = {
    # Simulation
    'MFSIM6': '73b086be-38f7-4c5d-942c-10007fb18b50',

    # Simulation children
    'TDIS6': '59ae8941-fade-4576-9d71-fb717c9999b3',
    'IMS6': '95426a1f-0409-49e9-b37b-dc66776aab52',

    # Exchanges
    'GWF6-GWF6': '54f91307-0b2d-41b2-aa60-1e7286194370',
    'GWF6-GWT6': 'b6d6323c-1d21-410a-913b-1ea898fa6862',
    'GWF6-GWE6': '0653b593-9e56-4da3-8018-e8bbc3ed9d84',
    'GWT6-GWT6': '05955a58-ece8-49d2-8d91-1072ccc50049',
    'GWE6-GWE6': '6c5c7bba-46e4-43a1-a4d8-6e197b4156de',

    # General packages
    'DIS6': 'f7604243-de9b-454f-b60d-2c47fde218b4',
    'DISV6': '8c74669d-5523-4eff-bf57-b74c880bc518',
    'DISU6': '4c0ab84c-0a49-43ef-bf01-8a83a6e31f68',
    'IC6': 'e84731ec-2047-4e6f-98be-81a597b55f7f',
    'OBS6': 'c1437fd6-8c12-487c-b461-34eb9b96e9b2',
    'OC6': '6600dedf-dab0-4b1c-aa30-99a968208b32',
    'MVR6': '18e1ee20-e343-4b43-be72-81c1140bf6bd',
    'POBS6': '51070b4f-3736-40ed-86f5-c19a1c6c4e25',

    # GWF
    'GWF6': '82dccf2d-51e4-4f42-a58a-3e0ca796e659',
    'BUY6': '1c919065-461c-4c4d-b136-573049ed86ce',
    'CHD6': '0f056089-1ec7-4b57-a4e8-279fac45bc78',
    'CSUB6': 'e2b4db65-27c3-4d6d-9030-e9ae49251faa',
    'DRN6': '10bd7fe0-0259-4b79-8c2a-c5d703e78274',
    'EVT6': '6b0ece31-fd82-4353-b2e4-86dbb62fa6b1',
    'EVTA6': 'f0177cc8-8d49-4e36-bca7-e7906d177a76',
    'GHB6': 'a2802cac-0233-4313-89f5-b694b51035c5',
    'GNC6': 'e25fb245-897b-40c1-a211-dff2354521fa',
    'HFB6': '3ecc6b62-ae09-4cc3-90c4-a6f8776cf8d4',
    'LAK6': '6e686288-3952-4314-b101-ee5d105bb633',
    'MAW6': 'b466eb76-8283-41e9-9b36-8f754ee368b3',
    'NPF6': 'b72b557b-f8a1-4bb1-9c74-045b1508631f',
    'RCH6': '45dfc4aa-b076-4c1d-a0c3-50a8466edf40',
    'RCHA6': '7cf7db7c-c0d1-41ce-a91a-0c4654a37023',
    'RIV6': '493a5038-3a25-4d4e-aceb-55da52957f8a',
    'SFR6': '90741996-83cf-423d-9d0d-460fe3d085f1',
    'STO6': '1a802c33-0450-465b-bc08-539993d1d001',
    'SWI6': '00065cb1-27bc-4754-a15f-0f7b1744ee1c',
    'UZF6': 'c6ebf6ab-eab0-4831-94f6-8ad7ed7b3e52',
    'VSC6': 'bf85d68d-73f2-4fe1-b9db-32395e5558d5',
    'WEL6': '9520a786-e678-4419-8646-a8f2f9d731f7',
    'ZONE6': 'a13177e8-230c-4975-aa9f-a0a3b906c6ef',

    # GWT
    'GWT6': 'a0b4991a-7f68-481f-9621-dc38cbb543cb',
    'ADV6': 'efc2e66a-dc63-4432-a68a-c62c06eba9e6',
    'CNC6': 'd311aca7-b8f7-4224-8970-df67cb0a274a',
    'DSP6': '359697c2-14ca-42ff-8070-fcd1213b3325',
    'FMI6': 'd6bc362a-86d2-4011-bd8d-acef44b2f112',
    'IST6': 'ea6fac2a-202a-4f1e-8a0b-9897da7be955',
    'LKT6': '5db81772-25b2-4242-93e4-b5488bb68f56',
    'MDT6': 'fec41085-c277-43f6-b957-34b27459423b',
    'MST6': 'afed7eff-5315-4e4a-a20f-1613ca3ccf3c',
    'MVT6': 'd0dfd8d2-50f6-4235-86d1-bbdb2235bf5f',
    'MWT6': 'e4c6e508-7381-4ebf-879f-50d0e7a2b308',
    'SFT6': '2d9f012a-fe53-45ab-8e37-a68567695609',
    'SRC6': 'd372a271-3309-47a5-bdc4-fde2376c0115',
    'SSM6': '9aa02b08-8c0c-4efc-8d4b-4e2939a4f4d2',
    'UZT6': '62ef0476-af29-4920-8f4a-31bfc1990b00',

    # GWE
    'GWE6': '929eef84-1901-4331-8b86-9445f415d311',
    'CND6': '87a06034-8334-4b4a-9649-3549cca11b13',
    'CTP6': '9a4e196a-bf3b-4fc5-9549-07be04c219d0',
    'ESL6': '67782a1a-c04d-4257-a21a-58d15a118dad',
    'EST6': '0afb3597-9f3d-4862-8bb8-be43aedb8e7a',
    'LKE6': 'dd15625b-80d1-468e-b23d-dad2aa08a9e2',
    'MVE6': 'a86fa2c3-d2dc-4613-8c71-fb336c85f206',
    'MWE6': 'db94f824-1631-4fba-b847-c47908f5f65a',
    'SFE6': 'a628231c-f98f-45e3-9de2-49c7306bec60',
    'UZE6': '625b60aa-4605-489c-a068-e44bf8987545',

    # Miscellaneous dialogs, in alphabetical order
    'AddStressesDialog': 'e2da7156-4c30-4f66-9ede-5deb4a281122',
    'ArraysToDatasetsDialog': '80beb999-42ef-4daf-b52b-1fb0c6e3e687',
    'CellPropertiesDialog': '530ead08-1306-4e09-bc42-b1e9749fe58b',
    'CopyPeriodDialog': '70c1644c-b62d-4c2b-83a0-c3ec1f91ba3e',
    # 'DateTimeDialog': '',  # NO_HELP_BUTTON
    # 'DefineStressPeriodDialog': '',  # NO_HELP_BUTTON
    'DisplayOptions': 'fb985b99-29f3-464d-a481-dd9ca554a9d0',
    'FlowBudgetDialog': '0f664f99-07f8-413d-b1eb-969e05b9bf04',
    # 'GetIntDialog': '',  # NO_HELP_BUTTON
    # 'GetPorosityDialog': '',  # NO_HELP_BUTTON
    'GeneratePestObsDialog': 'bc27811a-b4d2-4a76-aecf-bd2dea0340ed',
    'ListDialog': '88e95793-a166-4e74-ad0e-f545201fd64d',
    'MapFromCoverageDialog': 'c6494091-f62c-46d8-8780-705787c9ad0b',
    # 'MapFromShapefileDialog': '',  # Not done or released yet
    'NewSimDialog': 'c1b77616-85a9-4263-9a11-99e1e3c67a6f',
    'SwiSetUpDialog': 'f8b2942b-5f25-44ff-9d78-55961b9dcc62',
}


class SignalBlocker:
    """Context manager for QSignalBlocker, which apparently has its own in Pyside6."""
    def __init__(self, widget: QWidget):
        """Initializer.

        Args:
            widget: A widget.
        """
        self._widget = widget
        self._blocker: QSignalBlocker | None = None

    def __enter__(self):
        """Context manager enter.

        Returns:
            Self.
        """
        self._blocker = QSignalBlocker(self._widget)
        return self

    def __exit__(self, exc_type, exc_value, exc_tb) -> None:
        """Context manager exit.

        Args:
            exc_type: Exception type.
            exc_value: Exception value.
            exc_tb: Exception traceback.
        """
        self._blocker.unblock()


def handle_read_only_notice(locked, txt_read_only):
    """Shows or hides the read only message at the top of the dialog."""
    txt_read_only.setVisible(locked)
    if locked:
        txt_read_only.setStyleSheet("QLabel { background-color : rgb(255, 255, 128); color : rgb(0, 0, 0); }")


def open_file_in_default_app(filename, *args):
    """Opens the file in whatever app is associated with the file extension.

    *args exists because we pass this function as an argument to ListDialog,
    and we also sometimes pass ObsDialog.run_dialog_on_file, and it
    takes two args, so this one needs to also take two args.

    Args:
        filename (str): The filepath.
    """
    os.startfile(filename, 'open')


def set_vertical_header_menu_method(table, method):
    """Set the method to be called when right-clicking on the vertical header.

    Args:
        table: The table. Something derived from QTableView.
        method: Method to call when right-clicking in the header column.
    """
    table.verticalHeader().setContextMenuPolicy(Qt.CustomContextMenu)
    table.verticalHeader().customContextMenuRequested.connect(method)


def set_horizontal_header_menu_method(table, method):
    """Set the method to be called when right-clicking on the vertical header.

    Args:
        table: The table. Something derived from QTableView.
        method: Method to call when right-clicking in the header column.
    """
    table.horizontalHeader().setContextMenuPolicy(Qt.CustomContextMenu)
    table.horizontalHeader().customContextMenuRequested.connect(method)


def set_table_menu_method(table, method):
    """Set the method to be called when right-clicking in the table.

    Args:
        table: The table. Something derived from QTableView.
        method: Method to call when right-clicking anywhere else.
    """
    table.setContextMenuPolicy(Qt.CustomContextMenu)
    table.customContextMenuRequested.connect(method)


def get_selected_rows_or_warn(q_table):
    """Get the selected rows and warn the user if the selection is disjoint.

    If the selection is disjoint, warns the user and returns False.

    Returns:
        (tuple): tuple containing:
            - (bool): True or False.
            - (int): the minimum selected row.
            - (int): the maximum selected row.
    """
    selected_list = q_table.selectedIndexes()

    # Get the count of unique rows and the span from the selection
    unique_rows = get_unique_selected_rows(selected_list)
    if not unique_rows:
        return False, 0, 0
    minrow = min(unique_rows)
    maxrow = max(unique_rows)
    span = maxrow - minrow + 1

    if warn_if_disjoint_selection(q_table, len(unique_rows), span):
        return False, minrow, maxrow
    return True, minrow, maxrow


def get_unique_selected_rows(selected_list) -> list[int]:
    """Returns a list of selected row indices.

     Selection may be disjoint.

    Args:
        selected_list (list of QModelIndex): The selected list.

    Returns:
        See description.
    """
    row_set = set()
    for index in selected_list:
        row_set.add(index.row())
    return list(row_set)


def warn_if_disjoint_selection(parent, unique_rows, span):
    """Warns the user with a message box if unique_rows is not equal to span.

    Args:
        parent (Something derived from QWidget): The parent window.
        unique_rows (int): Number of unique rows selected.
        span (int): The maximum row selected minus the minimum row selected.

    Returns:
        bool: True if unique_rows != span, else False.
    """
    if unique_rows != span:
        message_box.message_with_ok(parent=parent, message='Cannot process disjoint selections.')
        return True
    return False


def read_csv_file_into_dataframe(
    filename, column_names, column_types, usecols: list | None = None, separator=io_util.mfsep
):
    """Reads the file into a pandas dataframe.

    Args:
        filename (str): Path of the external file.
        column_names (list of str): List of column names.
        column_types (dict of numpy number types) : Dictionary of column name and numpy type.
        usecols: pandas.read_csv() usecols arg. List of int or list of str. Causes other columns to be ignored.
        separator (str): Separator character.

    Returns:
         (DataFrame): The pandas DataFrame.
    """
    try:
        # start_time = time.time()

        # list_data_regex = r"\s*[,\s]\s*|[#!/]"  # Regex used to parse list data
        # # "Comment lines and blanks lines are also allowed within most blocks
        # # and within most input files. Valid comment characters include “#” “!”,
        # # and “//”. Comments can also be placed at the end of some input lines,
        # # after the required information. ... Comments included at the end of
        # # the line must be separated from the rest of the line by at least one
        # # space." mf6io.pdf page 3.
        # #
        # # "One or more spaces, or a single comma optionally combined with
        # #  spaces, must separate adjacent values." mf6io.pdf page 19.
        #
        # data_frame = pandas.read_csv(filename, header=None, names=column_names, index_col=False,
        #                              dtype=column_types, float_precision='high',
        #                              sep=list_data_regex, engine='python')

        # data_frame = pandas.read_csv(filename, header=None, names=column_names, index_col=False,
        #                              usecols=column_names, dtype=column_types, sep=io_util.mfsep,
        #                              float_precision='high')
        if filename:
            data_frame = pd.read_csv(
                filename,
                header=None,
                names=column_names,
                index_col=False,
                dtype=column_types,
                sep=separator,
                float_precision='high',
                quotechar='\'',
                usecols=usecols
            )
        else:
            data_frame = empty_dataframe(column_names, list(column_types.values()), index=None)

        data_frame.index = data_frame.index + 1  # Start index at 1, not 0
        # data_frame.fillna(value='', inplace=True)

        # print('Time to read from disk: {}'.format(time.time() - start_time))

    except Exception as error:
        raise error

    return data_frame


def default_dataframe(numeric_type, shape, default_value):
    """Creates a dataframe sized appropriately for the grid and DIS package.

    Args:
        numeric_type (str): 'int' or 'float'
        shape (GridInfo): dimensions of the grid
        default_value: The default value to assign to fill the dataframe with.

    Returns:
        dataframe (DataFrame): The pandas DataFrame.
    """
    dtype = np.int64 if numeric_type == 'int' else np.float64
    narray = np.empty(shape, dtype=dtype)
    narray.fill(default_value)

    data_frame = pd.DataFrame(data=narray)
    data_frame.index += 1
    data_frame.columns += 1
    return data_frame


def empty_dataframe(columns, dtypes, index=None):
    """Creates and returns an empty dataframe with the given column types.

    From https://stackoverflow.com/questions/36462257/create-empty-dataframe-in-pandas-specifying-column-types

    Args:
        columns (list of str): List of column names.
        dtypes (list of dtypes): List of column types.
        index: Optional index

    Returns:
        The new dataframe.
    """
    assert len(columns) == len(dtypes)
    df = pd.DataFrame(index=index)
    for c, d in zip(columns, dtypes):
        df[c] = pd.Series(dtype=d)
    return df


def change_columns_in_file(old_columns, new_columns, default_value, old_filename, new_filename):
    """Reads a file and writes a new file, fixing the columns from old_columns to new_columns.

    Args:
        old_columns (list or str): The columns in the old file.
        new_columns (list or str): The columns in the new file.
        default_value (str): Default value to add to new columns.
        old_filename (str): The old filename.
        new_filename (str): The new filename.
    """
    if not os.path.isfile(old_filename):
        return

    with open(old_filename, 'r') as infile, open(new_filename, 'w', newline='') as outfile:
        writer = csv.DictWriter(
            outfile, fieldnames=new_columns, restval=default_value, extrasaction='ignore', delimiter=io_util.mfsep
        )
        for row in csv.DictReader(infile, fieldnames=old_columns, delimiter=io_util.mfsep):
            writer.writerow(row)


def array_dataframe_to_temp_file(dataframe, array):
    """Writes the dataframe to disk and saves the filename into array.

    Also deletes any old file that may be saved with array.

    Args:
        dataframe (DataFrame): The DataFrame.
        array (Array): The array
    """
    temp_filename = dataframe_to_temp_file(dataframe)

    # Remove old temp file if one exists and update with new temp filename
    if array.temp_external_filename:
        os.remove(array.temp_external_filename)
    array.temp_external_filename = temp_filename


def dataframe_to_temp_file(dataframe, separator=io_util.mfsep):
    """Saves the dataframe to a temporary file and returns the filename.

    Args:
        dataframe (DataFrame): The dataframe.
        separator (str): Separator character.

    Returns:
        (str): The temporary filename.
    """
    # Create a new temp file
    dataframe.fillna(value=0.0, inplace=True)  # Otherwise nan becomes nothing and we are off when we read it back in
    with open(io_util.get_temp_filename(), mode='wt') as temp:
        dataframe_to_csv(dataframe, temp.name, separator=separator)
        temp.close()
        return temp.name


def dataframe_to_csv(df: pd.DataFrame, filepath: Pathlike, separator: str = io_util.mfsep, append: bool = False):
    """Wrapped DataFrame.to_csv() so we could use it with or without a file but with the same args.

    Args:
        df: The dataframe.
        filepath: Filepath. If empty, the csv string is returned.
        separator: Separator character.
        append: True to append to an existing file.

    Returns:
        (str): The csv string if filepath is empty, otherwise ''.
    """
    if filepath:
        mode = 'a' if append else 'w'
        df.to_csv(filepath, mode=mode, encoding='utf-8', index=False, header=False, sep=separator, quotechar='\'')
        return ''
    else:
        return df.to_csv(encoding='utf-8', index=False, header=False, sep=separator, quotechar='\'')


def apply_filter_on_selected_cells(dlg_input, grid_info, df):
    """Returns a dataframe filtered to only show selected cells, if there are any.

    Args:
        dlg_input (DialogInput): Information needed by the dialog.
        grid_info (GridInfo): Information about the grid.
        df: The dataframe.

    Returns:
        See description.
    """
    eval_str = ''
    if dlg_input.selected_cells and len(df.index) > 0:
        # with open('C:/temp/debug.txt', 'a') as file:
        #     file.write(f'grid_info: {grid_info}\n')
        gi = grid_info  # For convenience
        if dlg_input.data.ftype == 'SFR6':  # SFR now always has a CELLID column with 3, 2, or 1 integer in it
            if 'CELLIDX' not in df:
                # Append a CELLIDX column onto the dataframe
                cell_idxs = cell_idxs_from_cellids(grid_info, df['CELLID'].tolist())
                df['CELLIDX'] = cell_idxs
            return df[df['CELLIDX'].isin(dlg_input.selected_cells)]
        else:
            if gi.dis_enum == DisEnum.DIS and 'ROW' in df.columns:
                # eval_str = '((LAY - 1) * (@gi.nrow * @gi.ncol)) + ((ROW - 1) * @gi.ncol) + (COL - 1)'
                eval_str = f'((LAY - 1) * ({gi.nrow * gi.ncol})) + ((ROW - 1) * {gi.ncol}) + (COL - 1)'
            elif gi.dis_enum == DisEnum.DISV and 'CELL2D' in df.columns:
                # eval_str = '((LAY - 1) * @gi.ncpl) + CELL2D - 1'
                eval_str = f'((LAY - 1) * {gi.ncpl}) + CELL2D - 1'
            elif 'CELLID' in df.columns:
                eval_str = '(CELLID - 1)'

    if eval_str:
        cell_idxs = df.eval(eval_str)
        return df[cell_idxs.isin(dlg_input.selected_cells)]
    else:
        return df


def set_vertical_stretch(widget, stretch):
    """Sets the vertical stretch on the widget.

    Convenience function to avoid duplicating code.

    Args:
        widget: The widget.
        stretch (int): The stretch factor.
    """
    size = QSizePolicy(QSizePolicy.Preferred, QSizePolicy.Preferred)
    size.setVerticalStretch(stretch)
    widget.setSizePolicy(size)


def set_widget_fixed_width(widget: QWidget) -> None:
    """Used with QLineEdit and QSpinBox to set width so it doesn't grow wider.

    At least with numbers, they don't need to grow wider, and they make stuff look bad when they do. Calling
    setFixedWidth() seems to be only way that works.
    """
    min_width = 80  # This works well on my machine
    widget.setFixedWidth(max(min_width, widget.sizeHint().width()))  # Use size hint if it's wider


def setup_stress_period_spin_box_layout(parent, uix, nper, signal_method):
    """Sets up the stress period spinbox and create/delete toolbar.

    Combines common code in PeriodListWidget and PeriodArrayWidget.

    Args:
        parent: Parent widget.
        uix: Dict of ui items
        nper (int): Number of stress periods.
        signal_method: Method to call when the stress period changes.

    Returns:
        The layout.
    """
    # Create layout
    hlayout = QHBoxLayout()

    # Create label and add to layout
    lbl = uix['txt_period'] = QLabel('Period:')
    hlayout.addWidget(lbl)

    # Create spn control and add to layout
    spn = uix['spn_period'] = QSpinBox(parent)
    spn.setAccessibleName('Period')
    spn.setStyle(QxSpinBoxStyle())
    hlayout.addWidget(spn)
    spn.setMinimum(1)
    spn.setMaximum(nper)
    spn.setValue(1)
    spn.setKeyboardTracking(False)
    set_widget_fixed_width(spn)

    # Have spin control only signal when done editing
    spn.valueChanged.connect(signal_method, Qt.UniqueConnection)
    return hlayout


def set_label_styles_for_warning_message(qlabel):
    """Sets the label text and background color and other styles to indicate a warning.

    Args:
        qlabel (QLabel): The label.
    """
    qlabel.setWordWrap(True)
    qlabel.setOpenExternalLinks(True)
    qlabel.setTextFormat(Qt.RichText)
    qlabel.setTextInteractionFlags(Qt.TextBrowserInteraction)
    qlabel.setStyleSheet("QLabel { background-color : rgb(200, 200, 150); color : rgb(0, 0, 0); }")


def column_info_tuple_from_dict(columns):
    """Converts columns dict into a tuple with: name list, types dict, and defaults dict.

    Args:
        columns (dict): A dict of column names -> tuple with type and default.

    Returns:
        (tuple): tuple containing:
            - names (list): Column names.
            - types (dict of str -> type): Column names -> column types.
            - default (dict of str -> value): Column names -> default values.
    """
    names = []
    types = {}
    defaults = {}

    for key, item in columns.items():
        names.append(key)
        types[key] = item[0]
        defaults[key] = item[1]

    return names, types, defaults


def get_icon_path(node: TreeNode) -> str:
    """Method used by TreeItemSelectorDlg to display icons.

    Args:
        node (TreeNode): A tree node.

    Returns:
        (str): Icon path.
    """
    if node is None:
        return ''

    path = ''
    if node.item_typename == 'TI_COMPONENT':
        if node.unique_name == 'GWF6':
            path = ':/resources/icons/mf6_gwf_model.svg'
        elif node.unique_name == 'GWT6':
            path = ':/resources/icons/mf6_gwt_model.svg'
        elif node.unique_name == 'GWE6':
            path = ':/resources/icons/mf6_gwe_model.svg'
        elif node.unique_name == 'PRT6':
            path = ':/resources/icons/mf6_prt_model.svg'
        else:
            path = ':/resources/icons/package.svg'
    elif node.item_typename == 'TI_DYN_SIM_FOLDER':
        path = ':/resources/icons/mf6_simulations.svg'
    elif node.item_typename == 'TI_DYN_SIM':
        path = ':/resources/icons/mf6_simulation.svg'
    elif node.item_typename == 'TI_ROOT':
        if node.name == 'UGrid Data':
            path = ':/resources/icons/folder.svg'
    elif node.item_typename == 'TI_ROOT_SIMULATION':
        path = ':/resources/icons/simulation_data.svg'
    elif node.item_typename == 'TI_SCALAR_DSET':
        if node.name in {'Point Z', 'Cell Top Z', 'Cell Bottom Z'}:
            path = ':/resources/icons/elevation_data_inactive.svg'
        else:
            if node.data_location == 'CELL':
                path = ':/resources/icons/dataset_cells_inactive.svg'
            elif node.data_location == 'NODE':
                path = ':/resources/icons/dataset_points_inactive.svg'
    elif node.item_typename == 'TI_SOLUTION_FOLDER':
        path = ':/resources/icons/folder_locked.svg'
    # TreeNode doesn't have type of UGrid constraint so we can't do any better yet.
    # elif node.item_typename == 'TI_UGRID':
    #     path = ''
    return path


def set_read_only_and_grey(widget, on):
    """Sets the widget to be read-only and changes the color to grey, or the reverse.

    Args:
        widget: The widget.
        on (bool): Whether we are setting or unsetting.
    """
    widget.setReadOnly(on)
    if on:
        read_only_palette = QPalette()
        read_only_palette.setColor(QPalette.Base, widget.palette().color(QPalette.Window))
        widget.setPalette(read_only_palette)
    else:
        my_type = type(widget)
        default_widget = my_type()
        widget.setPalette(default_widget.palette())


def tab_index_from_text(tab_widget: QTabWidget, tab_text: str) -> int:
    """Returns the index of the tab with the given text, or -1 if not found.

    Args:
        tab_widget (QTabWidget): The tab widget.
        tab_text (str): Title of the tab in question.
    """
    tab_index = -1
    for index in range(tab_widget.count()):
        if tab_widget.tabText(0) == tab_text:
            tab_index = index
            break
    return tab_index


def _create_select_dataset_dialog(
    parent: QWidget, tree: TreeNode, selected_uuid: str, cell_count: int, ui_class=None
) -> TreeItemSelectorDlg:
    """Runs the dialog to let the user select a dataset and returns the selected dataset uuid, or ''.

    Args:
        parent: The parent widget.
        tree: The tree.
        selected_uuid: Uuid of thing to be preselected in the dialog.
        cell_count: Number of cells in the grid.
        ui_class: The ui class to use.

    Returns:
        See description.
    """
    def callback(node: TreeNode) -> bool:
        # Callback method to see if dataset should be shown in the dialog
        return node.num_vals == cell_count

    dialog = TreeItemSelectorDlg(
        title='Select Dataset',
        target_type=DatasetItem,
        pe_tree=tree,
        override_icon=get_icon_path,
        previous_selection=selected_uuid,
        parent=parent,
        selectable_callback=callback,
        ui_class=ui_class,
    )
    return dialog


def select_dataset_dialog(
    parent: QWidget,
    tree: TreeNode,
    selected_uuid: str,
    cell_count: int,
) -> None | str:
    """Runs the dialog to let the user select a dataset.

    Args:
        parent: The parent widget.
        tree: The tree.
        selected_uuid: Uuid of thing to be preselected in the dialog.
        cell_count: Number of cells in the grid.

    Returns:
        The uuid.
    """
    dialog = _create_select_dataset_dialog(parent, tree, selected_uuid, cell_count, None)
    if dialog.exec() == QDialog.Accepted:
        return dialog.get_selected_item_uuid()
    return None


def select_dataset_dialog_combo_box(
    parent: QWidget,
    tree: TreeNode,
    selected_uuid: str,
    cell_count: int,
    cbx_items: dict,
    cbx_start: str,
) -> tuple[str | None, Any]:
    """Runs the dialog to let the user select a dataset, adding a combo box at the top.

    Args:
        parent: The parent widget.
        tree: The tree.
        selected_uuid: Uuid of thing to be preselected in the dialog.
        cell_count: Number of cells in the grid.
        cbx_items: Dict of strings -> int to put in the combo box.
        cbx_start: Item to show in combo box to start.

    Returns:
        Tuple of uuid and QComboBox.currentData().
    """
    dialog = _create_select_dataset_dialog(
        parent, tree, selected_uuid, cell_count, Ui_dlg_treeitem_selector_with_combo_box()
    )
    dialog.ui.lbl_combobox.setText('Column:')
    for key, value in cbx_items.items():
        dialog.ui.cbx_combobox.addItem(key, value)
    dialog.ui.cbx_combobox.setCurrentText(cbx_start)
    if dialog.exec() == QDialog.Accepted:
        return dialog.get_selected_item_uuid(), dialog.ui.cbx_combobox.currentData()
    return None, None


def select_dataset_dialog_check_box(
    parent: QWidget, tree: TreeNode, selected_uuid: str, cell_count: int, chk_box_text: str,
    chk_box_state: Qt.CheckState
) -> tuple[str | None, Qt.CheckState]:
    """Runs the dialog to let the user select a dataset, adding a checkbox at the bottom.

    Args:
        parent: The parent widget.
        tree: The tree.
        selected_uuid: Uuid of thing to be preselected in the dialog.
        cell_count: Number of cells in the grid.
        chk_box_text: QCheckBox text.
        chk_box_state: Starting state of the QCheckBox.

    Returns:
        Tuple of uuid and QCheckBox.checkState().
    """
    dialog = _create_select_dataset_dialog(
        parent, tree, selected_uuid, cell_count, Ui_dlg_treeitem_selector_with_check_box()
    )
    dialog.ui.checkBox.setText(chk_box_text)
    dialog.ui.checkBox.setCheckState(chk_box_state)
    if dialog.exec() == QDialog.Accepted:
        return dialog.get_selected_item_uuid(), dialog.ui.checkBox.checkState()
    return None, None


def select_coverages_dialog(parent: QWidget, tree: TreeNode, selected_uuid: str) -> list[str]:
    """Run the TreeItemSelectorDlg to select coverages.

    Args:
        parent: The parent widget.
        tree: The tree.
        selected_uuid: Uuid of thing to be preselected in the dialog.

    Returns:
        List of the uuids selected.
    """
    dialog = TreeItemSelectorDlg(
        title='Select Coverage(s)',
        target_type=CoverageItem,
        pe_tree=tree,
        previous_selection=selected_uuid,
        parent=parent,
        allow_multi_select=True
    )
    uuids = []
    if dialog.exec() == QDialog.Accepted:
        uuids = dialog.get_selected_item_uuid()
    return uuids


def _correct_path(filepath: Path | str) -> str:
    """Return a valid path for open/save file dialogs: filepath if it exists, else an existing directory.

    Args:
        filepath: A filepath.

    Returns:
        See description.
    """
    filepath = Path(filepath)
    start_dir = Path(settings.get_file_browser_directory())
    if util.null_path(filepath):
        filepath = start_dir
    else:
        if io_util.is_filename(filepath):  # Just a file name?
            filepath = start_dir / filepath  # Add start_dir
        if not filepath.is_file():  # File doesn't exist?
            existing_parent = io_util.find_existing_parent(filepath.parent)
            if existing_parent:
                filepath = existing_parent / filepath.name  # Use an existing parent
            else:
                filepath = start_dir / filepath.name
    return str(filepath)


def run_open_file_dialog(
    parent: QWidget, caption: str, filepath: Path | str, filter_str: str, sel_filter: str = ''
) -> str:
    """Wrap QFileDialog.getOpenFileName(), adding getting existing dir and using get/set_file_browser_directory().

    Args:
        parent: The parent widget.
        caption: The dialog caption.
        filepath: The initial file path.
        filter_str: The filter string, e.g. 'CSV (Comma delimited) Files (*.csv);;All Files (*.*)'
        sel_filter: The selected filter.

    Return:
        The path selected from the dialog.
    """
    filepath_str = _correct_path(filepath)

    # Run the dialog
    filepath_str, _ = QFileDialog.getOpenFileName(parent, caption, filepath_str, filter_str, sel_filter)

    # Save the directory
    if filepath_str:
        dir_ = Path(filepath_str).parent
        settings.save_file_browser_directory(str(dir_))
    return filepath_str


def run_save_file_dialog(
    parent: QWidget, caption: str, filepath: Path | str, filter_str: str, sel_filter: str = ''
) -> str:
    """Wraps QFileDialog.getSaveFileName(), adding getting existing dir and using get/set_file_browser_directory().

    Args:
        parent: The parent widget.
        caption: The dialog caption.
        filepath: The initial file path.
        filter_str: The filter string, e.g. 'CSV (Comma delimited) Files (*.csv);;All Files (*.*)'
        sel_filter: The selected filter.

    Return:
        The path selected from the dialog.
    """
    filepath_str = _correct_path(filepath)

    # Run the dialog
    filepath_str, _ = QFileDialog.getSaveFileName(parent, caption, filepath_str, filter_str, sel_filter)

    # Save the directory
    if filepath_str:
        dir_ = Path(filepath_str).parent
        settings.save_file_browser_directory(str(dir_))
    return filepath_str


def help_getter(key: str) -> HelpGetter:
    """Return a HelpUrl set up using DialogInput.

    Args:
        key: The second part of the wiki help line on the above page (after the '|').

    Returns:
        See description.
    """
    return HelpGetter(key=key, default=util.wiki_mf6, dialog_help_url=util.wiki_dialog_help)


def new_table_view() -> QxTableView:
    """Return a new table view styled how we like it.

    Returns:
        The table view.
    """
    return widget_builder.new_styled_table_view(row_height=0)


def cell_idxs_from_cellids(grid_info: GridInfo, cellids: list) -> list[int]:
    """Return list of cell indices (0-based) given list of modflow cellids (can be tuples depending on DIS*).

    Args:
        grid_info: Grid information.
        cellids: List of cell ids, which can be ints (DISU), or list/tuples of ints (DIS, DISV).

    Returns:
        See description.
    """
    return [grid_info.cell_index_from_modflow_cellid([int(num) for num in cellid.split()]) for cellid in cellids]


def unique_name_no_spaces(names: set[str], name: str, sep_char: str = '_') -> str:
    """Return a name that is not in names and does not contain spaces.

    Args:
        names: A set of names.
        name: A candidate name.
        sep_char: Separator character, like '-' or '_'.

    Returns:
        See description.
    """
    # If name is good already, just return it
    if name not in names and ' ' not in name:
        return name

    # See if we can just replace spaces with dashes to make it unique
    if ' ' in name and name.replace(' ', sep_char) not in names:
        return name.replace(' ', sep_char)

    # Replace spaces with underscore
    name = name.replace(' ', sep_char)

    # Create regex that matches strings ending in sep_char followed by one or more digits, like 'bob-33'
    pattern = fr"{sep_char}\d+$"
    regex = re.compile(pattern)

    # Create unique name by having it end with char and a number
    while name in names:
        if regex.search(name):
            name = _increment_trailing_number(name, sep_char)
        else:
            name = f'{name}{sep_char}2'  # Append string with char and number to start the pattern
    return name


def _increment_trailing_number(text: str, sep_char: str) -> str:
    """Increments the number at the end of a string that follows an underscore.

    Example: "bob_33" becomes "bob_34"

    Args:
        text: The string to modify, e.g., "name_123".
        sep_char: The separator character, like '-' or '_'.

    Returns:
        The modified string with the incremented number, or the original
        string if the pattern isn't found.
    """
    # Pattern: Captures the number at the end preceded by the separator character.
    pattern = fr"{sep_char}(\d+)$"

    def replacer(match):
        """Replacement function for re.sub()."""
        number_str = match.group(1)
        incremented_number = int(number_str) + 1
        return f'{sep_char}{str(incremented_number)}'

    return re.sub(pattern, replacer, text)


def help_id_from_key(key: str) -> str:
    """Return the help id given a key.

    This function should be called for EVERY dialog that has a help button. Never should a help id be hardcoded
    anywhere else but here.

    Args:
        key: The key.
    """
    return _help_ids_dict.get(key, '')


def wrap_tip(tool_tip: str) -> str:
    """Return the tool tip string as rich text so it gets word wrapped.

     See https://stackoverflow.com/questions/4795757

    Args:
        tool_tip: The tool tip.

    Returns:
        See description.
    """
    return f'<span>{tool_tip}</span>'
