"""MvrDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.file_io import io_util
from xms.mf6.gui import gui_util
from xms.mf6.gui.options_gui import OptionsGui
from xms.mf6.gui.package_dialog_base import PackageDialogBase
from xms.mf6.gui.widgets.list_block_table_widget import ListBlockTableWidget
from xms.mf6.gui.widgets.period_list_widget import PeriodListWidget


class MvrDialog(PackageDialogBase):
    """A dialog used for the GNC package."""
    def __init__(self, dlg_input, parent=None):
        """Initializes the class, sets up the ui, and loads the package.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, parent)
        self.setup_ui()

    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections and self.default_sections should be set here.
        """
        self.sections = ['COMMENTS', 'OPTIONS', 'PACKAGES', 'PERIODS']
        self.default_sections = ['PERIODS']

    # @overrides
    def setup_options(self, vlayout):
        """Sets up the options section, which is defined dynamically, not in the ui file.

        Args:
            vlayout (QVBoxLayout): The layout that the option widgets will be added to.
        """
        self.options_gui = OptionsGui(self)
        self.options_gui.setup(vlayout)

    def setup_packages_section(self):
        """Sets up the packages section."""
        section = 'PACKAGES'
        self.add_group_box_to_scroll_area(section)

        # Table
        w = self.uix[section]['tbl'] = ListBlockTableWidget(self, self.dlg_input, '')
        self.uix[section]['layout'].addWidget(w)
        w.load_list_block('PACKAGES')

        # Make this section smaller than the periods section
        gui_util.set_vertical_stretch(self.uix[section]['group'], 1)

    def setup_periods_section(self):
        """Sets up the stress periods section."""
        section = 'PERIODS'
        self.add_group_box_to_scroll_area(section)

        w = self.uix[section]['sp_list_widget'] = PeriodListWidget(self.dlg_input.data, self.dlg_input, self)
        w.actions[':/resources/icons/plot.svg'].setVisible(False)  # Hide XY Series button. Not yet implemented.
        self.uix[section]['layout'].addWidget(self.uix[section]['sp_list_widget'])

        # Make this section larger than the packages section
        gui_util.set_vertical_stretch(self.uix[section]['group'], 2)

    # @overrides
    def get_table_widget(self, block):
        """Returns the table view from the given block.

        Args:
            block: The block.
        """
        return self.uix[block]['tbl']

    def setup_section(self, section_name):
        """Sets up a block of widgets.

        Args:
            section_name (str): name of the block
        """
        if section_name == 'PACKAGES':
            self.setup_packages_section()
        elif section_name == 'PERIODS':
            self.setup_periods_section()
        else:
            super().setup_section(section_name)

    def clean_up_temp_files(self):
        """Deletes all temp files."""
        io_util.clean_up_temp_files(self.dlg_input.data.period_files.values())

    def on_chk_modelnames(self, checked):
        """Called when the MODELNAMES checkbox is toggled.

        Args:
            checked (int): Qt.CHECKED etc
        """
        # Add or remove the mname, mname1, mname2 columns
        old_columns_packages, _, _ = self.dlg_input.data.get_column_info('PACKAGES')
        old_columns_periods, _, _ = self.dlg_input.data.get_column_info('')

        self.dlg_input.data.options_block.set('MODELNAMES', checked == Qt.Checked, None)

        new_columns_packages, _, _ = self.dlg_input.data.get_column_info('PACKAGES')
        new_columns_periods, _, _ = self.dlg_input.data.get_column_info('')

        table_widget = self.uix['PACKAGES']['tbl']
        table_widget.change_columns('PACKAGES', use_aux=False)

        sp_widget = self.uix['PERIODS']['sp_list_widget']
        sp_widget.change_columns(old_columns_periods, new_columns_periods, use_aux=False)

    @override
    def widgets_to_data(self) -> None:
        """Get info from widgets and store it in dlg_input.data."""
        super().widgets_to_data()
        if not self.dlg_input.locked:
            self.uix['PERIODS']['sp_list_widget'].accept()
            self.save_list_blocks_to_temp()

    def reject(self):
        """Called when the user clicks Cancel."""
        self.uix['PERIODS']['sp_list_widget'].reject()
        self.clean_up_temp_files()
        super().reject()
