"""TreeItemSelectorMapCoverage class."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass, field
from enum import Enum

# 2. Third party modules
from PySide2.QtWidgets import QDialog, QWidget

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import CoverageItem
from xms.api.tree import TreeNode
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.widgets import widget_builder
from xms.guipy.widgets.qx_file_picker_widget import adjust_path_to_fit, EventFilter

# 4. Local modules
from xms.mf6.file_io import io_util
from xms.mf6.gui import gui_util
from xms.mf6.gui.dialog_input import DialogInput
from xms.mf6.gui.map_from_coverage_dialog_ui import Ui_MapFromCoverageDialog
from xms.mf6.misc.settings import Settings


class MapOpt(Enum):
    """Append or replace enum."""
    APPEND = 0
    REPLACE = 1


@dataclass
class MapCoverageDlgInputs:
    """Inputs to the MapFromCoverageDialog."""
    pe_tree: TreeNode | None = None  # Root of the project explorer tree to display
    enable_append_replace: bool = True  # True to enable the append vs replace radio group
    append_or_replace: MapOpt = MapOpt.REPLACE
    previous_selection: list[str] = field(default_factory=list)  # UUID, or iterable, of the previous selection(s)
    override_layers: bool = False  # True to specify a .csv file used to override grid layers
    layer_filepath: str = ''  # Filepath to grid layer file
    parent: QWidget | None = None  # The parent window.
    allow_multi_select: bool = True


@dataclass
class MapCoverageDlgOutputs:
    """Outputs from the MapFromCoverageDialog."""
    item_uuids: list[str]  # UUIDs of the selection(s)
    append_or_replace: MapOpt
    override_layers: bool  # True to specify a .csv file used to override grid layers
    layer_filepath: str  # Filepath of layer file used to override grid layers


def run(inputs: MapCoverageDlgInputs) -> MapCoverageDlgOutputs | None:
    """Initializes the class, sets up the ui.

    Args:
        inputs: Dialog inputs.
    """
    dialog = MapFromCoverageDialog(inputs)
    if dialog.exec() == QDialog.Accepted:
        selected = dialog.get_selected_item_uuid()
        if not isinstance(selected, list):
            selected = [selected]
        rv = MapCoverageDlgOutputs(
            selected, MapOpt(dialog.get_replace()), dialog.get_override_layers(), dialog.get_layer_filepath()
        )
        return rv
    return None


def run_from_package(dlg_input: DialogInput, win_cont: QWidget) -> tuple[bool, MapCoverageDlgOutputs | None]:
    """Runs the tree selector dialog.

    Args:
        dlg_input: Information needed by the dialog.
        win_cont: The window container.

    Returns:
        (tuple): tuple containing:
            - (bool): True if OK and something selected, otherwise False.
            - (MapCoverageDlgOutputs): Outputs.
    """
    main_file = dlg_input.data.filename

    # Get inputs
    inputs = MapCoverageDlgInputs()
    inputs.pe_tree = dlg_input.query.project_tree
    inputs.append_or_replace = MapOpt(Settings.get(main_file, 'MAP_COVERAGE_REPLACE', 1))
    inputs.previous_selection = Settings.get(main_file, 'MAP_COVERAGE_UUIDS', [''])
    inputs.override_layers = bool(Settings.get(main_file, 'MAP_COVERAGE_OVERRIDE_LAYERS', False))
    inputs.layer_filepath = Settings.get(main_file, 'MAP_COVERAGE_LAYER_FILEPATH', '')
    inputs.parent = win_cont
    inputs.allow_multi_select = False if dlg_input.data.ftype == 'SFR6' else True

    # Determine append/replace
    inputs.enable_append_replace = True
    data_node = dlg_input.data.tree_node
    grid_ftypes = {'DIS6', 'DISV6', 'DISU6', 'STO6', 'MDT6', 'NPF6'}
    is_grid_type = data_node.unique_name in grid_ftypes
    readasarrays = data_node.unique_name in {'RCH6', 'EVT6'} and io_util.file_has_readasarrays(data_node.main_file)
    if is_grid_type or readasarrays:
        inputs.enable_append_replace = False
        inputs.append_or_replace = MapOpt.APPEND  # Always append for these guys.
    elif data_node.unique_name == 'SFR6':
        inputs.enable_append_replace = False
        inputs.append_or_replace = MapOpt.REPLACE

    # Run the dialog
    result = run(inputs)
    if not result:
        return False, None

    # Save the dialog settings to remember the next time
    if data_node.unique_name != 'SFR6' and not is_grid_type and not readasarrays:
        Settings.set(main_file, 'MAP_COVERAGE_REPLACE', result.append_or_replace.value)
    Settings.set(main_file, 'MAP_COVERAGE_UUIDS', result.item_uuids)
    Settings.set(main_file, 'MAP_COVERAGE_OVERRIDE_LAYERS', result.override_layers)
    Settings.set(main_file, 'MAP_COVERAGE_LAYER_FILEPATH', result.layer_filepath)
    return True, result


class MapFromCoverageDialog(TreeItemSelectorDlg):
    """Dialog for the Map from Coverage command."""
    def __init__(self, inputs: MapCoverageDlgInputs):
        """Initializes the class, sets up the ui.

        Args:
            inputs: Dialog inputs.
        """
        super().__init__(
            title='Select Coverage',
            target_type=CoverageItem,
            pe_tree=inputs.pe_tree,
            previous_selection=inputs.previous_selection,
            parent=inputs.parent,
            ui_class=Ui_MapFromCoverageDialog(),
            allow_multi_select=inputs.allow_multi_select
        )
        self._inputs = inputs
        self._event_filter = None
        self._layer_filepath = ''

        # Initialize gui
        self.help_getter = gui_util.help_getter(gui_util.help_id_from_key('MapFromCoverageDialog'))
        self._init_append_or_replace_radio_group()
        self._init_override_layers()

        # Signals
        self.ui.buttonBox.helpRequested.connect(self.help_requested)
        self.ui.btn_browse.clicked.connect(self._on_browse)
        self.ui.chk_override_layers.stateChanged.connect(self._do_enabling)

        self._do_enabling()

    def _do_enabling(self) -> None:
        """Enables/disables things."""
        # Append/replace radio group
        self.ui.rbt_append.setEnabled(self._inputs.enable_append_replace)
        self.ui.rbt_replace.setEnabled(self._inputs.enable_append_replace)

        # Override layers
        self.ui.edt_layer_file.setEnabled(self.ui.chk_override_layers.isChecked())
        self.ui.btn_browse.setEnabled(self.ui.chk_override_layers.isChecked())

    def _init_append_or_replace_radio_group(self) -> None:
        """Initialize the append/replace radio group."""
        if self._inputs.append_or_replace == MapOpt.APPEND:
            self.ui.rbt_append.setChecked(True)
        else:
            self.ui.rbt_replace.setChecked(True)

    def _init_override_layers(self) -> None:
        """Initialize the override layers widgets."""
        self.ui.chk_override_layers.setChecked(self._inputs.override_layers)
        widget_builder.make_lineedit_readonly(self.ui.edt_layer_file)
        self._event_filter = EventFilter(self)  # To handle resize events and adjust path
        self.installEventFilter(self._event_filter)
        self.layer_filepath = self._inputs.layer_filepath  # Calls setter, which may adjust displayed path to fit

    @property
    def layer_filepath(self) -> str:
        """Return the override grid layers file path.

        Returns:
            The filepath.
        """
        return self._layer_filepath

    @layer_filepath.setter
    def layer_filepath(self, path: str) -> None:
        """Set the override grid layers file path.

        Args:
            path: The filepath.
        """
        self._layer_filepath = path
        self.ui.edt_layer_file.setText(adjust_path_to_fit(path, self.ui.edt_layer_file))

    def on_resize(self):
        """Called when self.ui.edt_layer_file is resized."""
        self.layer_filepath = self._layer_filepath  # Trigger code that may shorten the path

    def _on_browse(self) -> None:
        """Called when the browse button is clicked."""
        filter_str = 'CSV (Comma delimited) Files (*.csv)'
        start_filepath = self._inputs.layer_filepath
        filepath = gui_util.run_open_file_dialog(self, 'Grid Layer File', start_filepath, filter_str, filter_str)
        if not filepath:
            return
        self.layer_filepath = filepath

    def get_replace(self) -> int:
        """Returns 0 if appending, 1 if replacing.

        Returns:
            See description.
        """
        return int(self.ui.rbt_replace.isChecked())

    def get_override_layers(self) -> bool:
        """Return True if overriding grid layers, else False.

        Returns:
            See description.
        """
        return self.ui.chk_override_layers.isChecked()

    def get_layer_filepath(self) -> str:
        """Return the grid layer filepath.

        Returns:
            See description.
        """
        return self.layer_filepath

    def accept(self):
        """Don't let the user click OK without having selected something."""
        if not self.get_selected_item_uuid():
            message_box.message_with_ok(parent=self, message='A coverage must be selected to continue.')
            return
        super().accept()
