"""Module for the PtmRunManager class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['PtmModelRunManager']

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.components.dmi.model_run_manager import ModelRunManager
from xms.components.dmi.xms_data import XmsData
from xms.gmi.data_bases.sim_base_data import SimBaseData
from xms.guipy.dialogs.process_feedback_dlg import run_feedback_dialog
from xms.guipy.time_format import string_to_datetime

# 4. Local modules
from xms.ptm.entry_points import model_run, model_run_tracker
from xms.ptm.feedback.load_solution_thread import PtmLoadSolutionThread
from xms.ptm.model.model import simulation_model, StopMode


class PtmModelRunManager(ModelRunManager):
    """Class that finds executables for running PTM."""
    def find_model_run_script(self, file_location: str):
        """
        Find the model run and progress scripts.

        Args:
            file_location: Folder where XMS thinks the model native files should be.
        """
        control_file = f'{self.data.simulation_name}.pcf'
        control_path = Path(file_location) / control_file
        if not control_path.exists():
            self.report_error(
                f'Unable to run PTM because "{control_path}" was not found. Ensure the simulation has been exported.'
            )
            return

        model_path = self.find_model_executable('PTM Executable')
        if not model_path:
            self.report_error(
                'Unable to run PTM because the model executable was not set. '
                'Check the "PTM Executable" file location in the SMS Preferences.'
            )
            return
        elif not Path(model_path).exists():
            self.report_error(
                'Unable to run PTM because the model executable does not exist. '
                'Check the "PTM Executable" file location in the SMS Preferences.'
            )
            return
        elif not Path(model_path).is_file():
            self.report_error(
                'Unable to run PTM because the model executable is not a file. '
                'Check the "PTM Executable" file location in the SMS Preferences.'
            )
            return

        num_time_steps = self.get_num_time_steps()

        self.runner = model_run.__file__
        self.runner_args.append(str(model_path))
        self.runner_args.append(str(control_path))

        self.tracker = model_run_tracker.__file__
        self.tracker_args.append(str(num_time_steps))

    def get_num_time_steps(self) -> int:
        """Find how many time steps the simulation is expected to run for."""
        sim_data: SimBaseData = self.data.simulation_data
        section = simulation_model()
        section.restore_values(sim_data.global_values)

        group = section.group('time')
        if group.parameter('stop_mode').value == StopMode.duration:
            interval = group.parameter('duration').value
        else:
            start_date = string_to_datetime(group.parameter('start_run').value)
            end_date = string_to_datetime(group.parameter('stop_run').value)
            interval = (end_date - start_date).total_seconds()

        time_step = group.parameter('time_step').value
        time_steps = int(interval / time_step)
        return time_steps

    def load_solution(self, data: XmsData, parent: QWidget):
        """Called by XMS when it wants to load the solution."""
        thread = PtmLoadSolutionThread(data)
        run_feedback_dialog(thread, parent=parent)
