"""Module for ImportSimThread class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['ImportSimThread']

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.component_builders.main_file_maker import create_component
from xms.components.dmi.xms_data import XmsData
from xms.guipy.dialogs.feedback_thread import FeedbackThread

# 4. Local modules
from xms.ptm.components.sim_component import PtmSimComponent
from xms.ptm.file_io.control import read_control
from xms.ptm.file_io.sources.source_reader import read_sources
from xms.ptm.file_io.traps.trap_reader import read_traps


class ImportSimThread(FeedbackThread):
    """Import thread."""
    def __init__(self, data: XmsData):
        """
        Construct the worker.

        Args:
            data: Interprocess communication object.
        """
        super().__init__(xms_data=data)
        self.display_text |= {
            'title': 'Reading PTM Control File',
            'working_prompt': 'Reading control file. Please wait...',
        }
        self._data = data
        self._source_file: str = ''
        self._trap_file: str = ''

    def _run(self):
        """Run the thread."""
        self._log.info('Importing simulation...')
        self._read_control()
        self._read_sources()
        self._read_traps()

    def _read_control(self):
        """Read the control file."""
        global_parameters = read_control(self._data.file_to_read)
        sim_component = create_component(PtmSimComponent)
        sim_component.data.global_values = global_parameters.extract_values()
        sim_component.data.commit()

        name = self._data.file_to_read.stem
        self._data.add_simulation(sim_component, name)

        self._source_file = global_parameters.group('files').parameter('source_file').value
        self._trap_file = global_parameters.group('files').parameter('trap_file').value

    def _read_sources(self):
        """Read the Source file and build the Source coverage, if necessary."""
        if not self._source_file:
            return

        source_file = Path(self._source_file).absolute()
        if not source_file.exists():
            self._log.warning(f'The file "{source_file}" does not exist. Sources will not be read.')
            return

        coverage, component = read_sources(self._source_file)
        self._data.add_coverage(coverage, component)
        self._data.link_item(coverage.uuid)

    def _read_traps(self):
        """Read the trap file and build its coverage, if necessary."""
        if not self._trap_file:
            return

        trap_file = Path(self._trap_file).absolute()
        if not trap_file.exists():
            self._log.warning(f'The file "{trap_file}" does not exist. Traps will not be read.')
            return

        coverage, component = read_traps(self._trap_file)
        self._data.add_coverage(coverage, component)
        self._data.link_item(coverage.uuid)
