"""Cell monitor data."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import html

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel
from xms.tool_core.table_definition import FloatColumnType, TableDefinition

# 4. Local modules
from xms.rsm.data import bc_val_data_def as bvdd
from xms.rsm.data import monitor_data_def as mdd
from xms.rsm.data.mesh_data_def import add_rain_et_to_group

HPM_LITZONE = 'litzone'
HPM_PAN_CONST = 'pan-constantArea'
HPM_PAN_VAR = 'pan-variableArea'
HPM_SFWMM = 'sfwmmSTA'
HPM_OPTS = [HPM_LITZONE, HPM_PAN_CONST, HPM_PAN_VAR, HPM_SFWMM]


class _LakeDataDef:
    """A class to define the lake data for the generic model."""
    def __init__(self):
        self.generic_model = GenericModel(exclusive_polygon_conditions=False)
        self._pp = self.generic_model.polygon_parameters
        self._polygon_data_def()

    def _polygon_data_def(self):
        """Defines the polygon data for the generic model."""
        self._add_lake_group()
        # rain/et
        rain_gp = self._pp.add_group(group_name='rain', label='Lake rain')
        et_gp = self._pp.add_group(group_name='refet', label='Lake evapotranspiration')
        add_rain_et_to_group(rain_gp, et_gp)
        self._add_hpm_group()  # lake HPM
        self._add_bc_group()
        self._monitor()

    def _add_lake_group(self):
        """Adds the lake group to the polygon parameters."""
        gp = self._pp.add_group(group_name='lake', label='Lake')
        gp.add_text(name='label', label='Lake label (<label>)')
        gp.add_float(name='head0', label='Initial lake head (<head0>)', default=0.0)
        opts = ['SSTable', 'polynomial', 'cylinder', 'parabolic']
        opt_par = gp.add_option(name='package', label='Storage package', default=opts[0], options=opts)
        # inputs for SSTable
        cols = [FloatColumnType(header='Stage', default=0.0), FloatColumnType(header='Volume', default=0.0)]
        td = TableDefinition(cols)
        default_vals = []
        p = gp.add_table(name='sv', label='Stage-volume table (<sv>)', table_definition=td, default=default_vals)
        flags = {o: False for o in opts}
        flags[opts[0]] = True
        p.add_dependency(parent=opt_par, flags=flags)
        cols = [FloatColumnType(header='Stage', default=0.0), FloatColumnType(header='Area', default=0.0)]
        td = TableDefinition(cols)
        p = gp.add_table(name='sa', label='Stage-area table (<sa>)', table_definition=td, default=default_vals)
        p.add_dependency(parent=opt_par, flags=flags)
        # polynomial inputs
        flags = {o: False for o in opts}
        flags[opts[1]] = True
        p = gp.add_float(name='hmin', label='Minimum head in fitted polynomial (<hmin>)', default=0.0)
        p.add_dependency(parent=opt_par, flags=flags)
        p = gp.add_float(name='hmax', label='Maximum head in fitted polynomial (<hmax>)', default=0.0)
        p.add_dependency(parent=opt_par, flags=flags)
        cols = [FloatColumnType(header='Stage-area coefficients', default=0.0)]
        td = TableDefinition(cols, fixed_row_count=5)
        label = html.escape('Stage-area polynomial coefficients (<A>)')
        p = gp.add_table(name='poly_a', label=label, table_definition=td, default=default_vals)
        p.add_dependency(parent=opt_par, flags=flags)
        cols = [FloatColumnType(header='Stage-storage coefficients', default=0.0)]
        td = TableDefinition(cols, fixed_row_count=5)
        label = html.escape('Stage-storage polynomial coefficients (<B>)')
        p = gp.add_table(name='poly_b', label=label, table_definition=td, default=default_vals)
        p.add_dependency(parent=opt_par, flags=flags)
        # cylinder inputs
        flags = {o: False for o in opts}
        flags[opts[2]] = True
        p = gp.add_float(name='bottom', label='Minimum lake elevation (<bot>)', default=0.0)
        p.add_dependency(parent=opt_par, flags=flags)
        p = gp.add_float(name='toparea', label='Surface area at top elevation (<toparea>)', default=0.0)
        p.add_dependency(parent=opt_par, flags=flags)
        # parabolic inputs
        flags = {o: False for o in opts}
        flags[opts[3]] = True
        p = gp.add_float(name='top', label='Lake elevation at top of parabolic portion (<top>)', default=0.0)
        p.add_dependency(parent=opt_par, flags=flags)
        p = gp.add_float(name='para_bottom', label='Lake bottom elevation (<bot>)', default=0.0)
        p.add_dependency(parent=opt_par, flags=flags)
        p = gp.add_float(name='para_toparea', label='Surface area at top elevation (<toparea>)', default=0.0)
        p.add_dependency(parent=opt_par, flags=flags)

    def _add_hpm_group(self):
        """Adds the HPM group to the polygon parameters."""
        hpm_gp = self._pp.add_group(group_name='hpm', label='Lake HPM')
        hpm_gp.add_text(name='label', label='Label (<label>)')
        hpm_gp.add_text(name='tag', label='HPM output tag (<tag>)')
        opts = HPM_OPTS
        opt_par = hpm_gp.add_option(name='package', label='HPM package', default=opts[0], options=opts)
        hpm_gp.add_text(name='package_label', label='Package label (<label>)')
        # litzone inputs
        flags = {o: False for o in opts}
        flags[opts[0]] = True
        label = html.escape('Open water coefficient (<A>)')
        p = hpm_gp.add_float(name='open_water_coef', label=label, default=0.0)
        p.add_dependency(parent=opt_par, flags=flags)
        label = html.escape('Littoral zone ET coefficient (<A>)')
        p = hpm_gp.add_float(name='et_coef', label=label, default=0.0)
        p.add_dependency(parent=opt_par, flags=flags)
        label = html.escape('Monthly ET monitor (<B>)')
        p = hpm_gp.add_text(name='monthly_et', label=label, default='')
        p.add_dependency(parent=opt_par, flags=flags)
        p = hpm_gp.add_text(name='monthly_rain', label='Monthly rain monitor (<C>)', default='')
        p.add_dependency(parent=opt_par, flags=flags)
        cols = [FloatColumnType(header='Stage', default=0.0), FloatColumnType(header='Area', default=0.0)]
        td = TableDefinition(cols)
        default_vals = []
        p = hpm_gp.add_table(name='sa', label='Stage-area table (<sa>)', table_definition=td, default=default_vals)
        p.add_dependency(parent=opt_par, flags=flags)
        # pan-constantArea inputs
        flags = {o: False for o in opts}
        flags[opts[1]] = True
        p = hpm_gp.add_float(name='pca_et_coef', label='ET coefficient (<p1>)', default=0.0)
        p.add_dependency(parent=opt_par, flags=flags)
        p = hpm_gp.add_float(name='pca_area', label='Interception area (acres) (<p2>)', default=0.0)
        p.add_dependency(parent=opt_par, flags=flags)
        # pan-variableArea inputs
        flags = {o: False for o in opts}
        flags[opts[2]] = True
        p = hpm_gp.add_float(name='pva_et_coef', label='ET coefficient (<p1>)', default=0.0)
        p.add_dependency(parent=opt_par, flags=flags)
        # sfwmmSTA inputs
        flags = {o: False for o in opts}
        flags[opts[3]] = True
        p = hpm_gp.add_float(name='kmax', label='Open water crop coefficient (<kmax>)', default=0.0)
        p.add_dependency(parent=opt_par, flags=flags)
        p = hpm_gp.add_float(name='owpond', label='Open water depth (<owpond>)', default=0.0)
        p.add_dependency(parent=opt_par, flags=flags)
        p = hpm_gp.add_float(name='drz', label='Deep root zone depth (ft) (<drz>)', default=0.0)
        p.add_dependency(parent=opt_par, flags=flags)
        p = hpm_gp.add_float(name='srz', label='Shallow root zone depth (ft) (<srz>)', default=0.0)
        p.add_dependency(parent=opt_par, flags=flags)
        p = hpm_gp.add_text(name='kcoefID', label='Crop coefficient rule curve (<kcoefID>)', default='')
        p.add_dependency(parent=opt_par, flags=flags)

    def _add_bc_group(self):
        """Adds the HPM group to the polygon parameters."""
        bc_gp = self._pp.add_group(group_name='bc', label='Lake Boundary Conditions')
        bc_gp.add_text(name='label', label='Label (<label>)')
        bc_gp.add_integer(name='bc_id', label='Boundary condition id (<id>)', default=-1)
        opts = ['lakesource', 'owet', 'lakeHeadBC', 'lakeghb']
        opt_par = bc_gp.add_option(name='bc_type', label='BC type', default=opts[0], options=opts)
        # lakesource
        flags = {o: False for o in opts}
        flags[opts[0]] = True
        pkg_opts = ['Not specified', 'areaCorrected']
        label = 'Lake source package'
        opt_pkg = bc_gp.add_option(name='lksrc_pkg', label=label, default=pkg_opts[0], options=pkg_opts)
        opt_pkg.add_dependency(parent=opt_par, flags=flags)
        flags2 = {o: False for o in pkg_opts}
        flags2[pkg_opts[1]] = True
        p = bc_gp.add_float(name='lksrc_area', label='Lake source area (ft^2)', default=0.0)
        p.add_dependency(parent=opt_pkg, flags=flags2)
        # owet
        flags = {o: False for o in opts}
        flags[opts[1]] = True
        cols = [FloatColumnType(header='Stage', default=0.0), FloatColumnType(header='Area', default=0.0)]
        td = TableDefinition(cols)
        default_vals = []
        p = bc_gp.add_table(name='sa', label='Stage-area table (<sa>)', table_definition=td, default=default_vals)
        p.add_dependency(parent=opt_par, flags=flags)
        # lakeHeadBC - nothing to add here
        # lakeghb
        flags = {o: False for o in opts}
        flags[opts[3]] = True
        ghb_opts = ['both', 'onlyOut', 'onlyIn']
        lab = 'GHB flow direction (<flowdir>)'
        p = bc_gp.add_option(name='flowdir', label=lab, default=ghb_opts[0], options=ghb_opts)
        p.add_dependency(parent=opt_par, flags=flags)
        p = bc_gp.add_float(name='kcoeff', label='GHB coefficient (<kcoeff>)', default=0.0)
        p.add_dependency(parent=opt_par, flags=flags)
        # add the boundary condition values
        bvdd.add_bc_val_to_group(group=bc_gp)

    def _monitor(self):
        pp = self._pp
        outputs = [
            ('lakesto', 'Monitor storage volume (<lakesto>)'),
            ('initlakesto', 'Monitor initial storage volume (<initlakesto>)'),
            ('deltasto', 'Monitor delta storage (<deltasto>)'),
            ('head', 'Monitor head (<head>)'),
            ('head_iter', 'Monitor head at each iteration (<head_iter>)'),
            ('depth', 'Monitor depth of lake (<depth>)'),
            ('recharge', 'Monitor recharge volume (<recharge>)'),
            ('runoff', 'Monitor runoff volume received (<runoff>)'),
            ('wsupply', 'Monitor water supply withdrawal volume (<wsupply>)'),
            ('lakerecharge', 'Monitor lake recharge volume (<lakerecharge>)'),
            ('lakerunoff', 'Monitor lake runoff received (<lakerunoff>)'),
            ('lakewsupply', 'Monitor lake water supply withdrawals (<lakewsupply>)'),
        ]
        for name, label in outputs:
            gp = pp.add_group(group_name=name, label=label)
            mdd.add_output_to_group(gp)


def generic_model():
    """Returns a generic model with the definition for polygon properties.

    Returns:
        (GenericModel): see above
    """
    ldd = _LakeDataDef()
    return ldd.generic_model
