"""RuleCurveData class."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.bases.visible_coverage_component_base_data import VisibleCoverageComponentBaseData
from xms.gmi.data.generic_model import GenericModel
from xms.gmi.data_bases.coverage_base_data import CoverageBaseData

# 4. Local modules
import xms.rsm.data.rule_curve_data_def as rcdd


class RuleCurveData(CoverageBaseData, VisibleCoverageComponentBaseData):
    """Manages data file for a RuleCurveComponent."""
    def __init__(self, main_file: str | Path):
        """
        Initialize the data class.

        Args:
            main_file: The netcdf file (with path) associated with this instance data. Probably the owning
                       component's main file. If not provided, a default one will be chosen.
        """
        super().__init__(main_file)
        self.info.attrs['FILE_TYPE'] = 'RSM_RULE_CURVE_COV'

    @property
    def generic_model(self) -> GenericModel:
        """
        The generic model.

        - This only gets the model's template. Values are not included.
        - This is mainly useful for models with multiple or dynamic templates. Derived models should typically just have
          a function that makes their model template and always use that to get it.
        """
        definitions = self.info.attrs.get('GENERIC_MODEL_TEMPLATE', None)
        if definitions is None:
            gm = rcdd.generic_model()
        else:
            gm = GenericModel(definitions=definitions)
        return gm

    @generic_model.setter
    def generic_model(self, generic_model: GenericModel):
        """
        The generic model.

        This only sets the model's template. Values are not included.
        """
        self.info.attrs['GENERIC_MODEL_TEMPLATE'] = generic_model.to_template().to_string()

    @property
    def values(self) -> str:
        """
        The shared values for material parameters.

        This does not include group activity.
        """
        if 'RULE_CURVE_VALUES' in self.info.attrs:
            return self.info.attrs['RULE_CURVE_VALUES']
        # Let GenericModel define the defaults in case it wants to change them.
        return GenericModel().material_parameters.extract_values()

    @values.setter
    def values(self, value: str):
        """
        The shared values for material parameters.

        This does not include group activity.
        """
        self.info.attrs['RULE_CURVE_VALUES'] = value
