"""Class for writing mesh bcs to model input files."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import xml.etree.cElementTree as Et

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.rsm.data import mesh_data_def as mdd
from xms.rsm.data.mesh_property_data import CoverageMeshPropertyData
from xms.rsm.file_io.prop_const_dataset_writer import PropConstDatasetWriter
from xms.rsm.file_io.rain_et_writer import RainETWriter


class WaterBodyRainEtData:
    """Class for holding lake rain/ET data."""
    def __init__(
        self, xms_data, group_name, parent_xml, polygon_parameters, csv_writer, water_body_name, poly_id, cov_name
    ):
        """Constructor.

        Args:
            xms_data (XmsData): Simulation data retrieved from SMS
            group_name (str): Name of lake group
            parent_xml (xml.etree.cElementTree.Element): xml element for the lake
            polygon_parameters (xms.gmi.generic_model.Section): polygon parameters section
            csv_writer (CsvWriter): csv writer for writing csv files
            water_body_name (str): name of the water body (e.g., 'Lake', 'Basin')
            poly_id (int): polygon id
            cov_name (str): name of the coverage
        """
        self.xms_data = xms_data
        self.group_name = group_name
        self.parent_xml = parent_xml
        self.pp = polygon_parameters
        self.csv_writer = csv_writer
        self.water_body_name = water_body_name
        self.cur_poly_id = poly_id
        self.cov_name = cov_name


class WaterBodyRainEtWriter:
    """Class for writing lake rain/ET data to the xml."""
    def __init__(self, data):
        """Constructor.

        Args:
            group_name (str): the group name for the rain/ET data (e.g., 'rain', 'refet')
            data (WaterBodyRainEtData): the water body data object containing the necessary information
            water_body_name (str): name of the water body for error messages
        """
        self.data = data
        self.prop_name = f'{self.data.water_body_name} rain'
        self._mp = None
        self._poly_id_str = ''

    def write(self):
        """Write the rain or refet data to the xml."""
        if self.data.group_name == 'refet':
            self.prop_name = f'{self.data.water_body_name} evapotranspiration'
        self._poly_id_str = f'({self.data.cur_poly_id}, {self.data.cov_name})'
        lmp = CoverageMeshPropertyData(self.data.pp)
        self._mp = lmp.mesh_property_data(self.data.group_name)
        if self._mp.option != mdd.OPT_NOT_SPECIFIED:
            elem = Et.SubElement(self.data.parent_xml, self.data.group_name)
            self._rain_et_write(elem)

    def _rain_et_write(self, elem):
        """Write the lake rain/ET data to the xml."""
        if self._mp.option in [mdd.OPT_CONSTANT, mdd.OPT_DATASET]:
            self._rain_et_const_dataset(elem)
        else:
            self._rain_et_other_opts(elem)

    def _rain_et_const_dataset(self, elem):
        writer = PropConstDatasetWriter(
            self.data.xms_data,
            self.prop_name,
            self._mp.option,
            self._mp.const,
            'const',
            'value',
            self._mp.dataset,
            'gms',
            elem,
            self._poly_id_str,
        )
        writer.write()

    def _rain_et_other_opts(self, elem):
        """Write the rain/ET data for other options."""
        prop = self.prop_name.replace(' ', '_')
        csv_name = f'{prop}_{self.data.cov_name}_{self.data.cur_poly_id}'
        writer = RainETWriter(
            option=self._mp.option,
            rain_et=self._mp.rain_et,
            xml_parent=elem,
            csv_writer=self.data.csv_writer,
            csv_fname=csv_name,
            poly_ids_cov=self._poly_id_str,
        )
        writer.write()
