"""MonitorData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import json

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.srh.data.par.monitor_arc_param import MonitorArc
from xms.srh.data.par.monitor_point_param import MonitorPoint
from xms.srh.data.srh_coverage_data import SrhCoverageData


class MonitorData(SrhCoverageData):
    """Class for storing the SRH Material properties (Manning's N and sediment)."""
    def __init__(self, filename):
        """Constructor.

        Args:
            filename (:obj:`str`): file name
        """
        super().__init__(filename, 'SRH_MONITOR_DATA', 'point_data')

    @property
    def monitor_df(self):
        """Get the dataframe of the monitor data."""
        return self.data.to_dataframe()

    def next_comp_id(self):
        """Gets the next component id.

        Returns:
            (:obj:`int`): The next component id
        """
        df = self.data.to_dataframe()
        return int(max(df['id'])) + 1

    def monitor_point_param_from_id(self, data_id):
        """Gets a record from an id. If the id is not in the point_data_asdf then returns None.

        Args:
            data_id (:obj:`int`): component id

        Returns:
            (:obj:`PointParam`): The record from the bc dataframe as a PointParam class
        """
        obs = MonitorPoint()
        return self.param_from_id(data_id, obs)

    def monitor_arc_param_from_id(self, data_id):
        """Gets a record from an id. If the id is not in the point_data_asdf then returns None.

        Args:
            data_id (:obj:`int`): component id

        Returns:
            (:obj:`PointParam`): The record from the bc dataframe as a PointParam class
        """
        obs = MonitorArc()
        return self.param_from_id(data_id, obs)

    def append_point_data_with_id(self, monitor_point_data, data_id):
        """Sets a record with an id. If the id is < 1 then do nothing.

        Args:
            monitor_point_data (:obj:`MonitorPoint`): instance of monitor point data
            data_id (:obj:`int`): component id

        """
        self.append_param_data_with_id(monitor_point_data, data_id)

    def append_arc_data_with_id(self, monitor_arc_data, data_id):
        """Sets a record with an id. If the id is < 1 then do nothing.

        Args:
            monitor_arc_data (:obj:`MonitorArc`): instance of monitor arc data
            data_id (:obj:`int`): component id

        """
        self.append_param_data_with_id(monitor_arc_data, data_id)

    def get_arc_comp_ids_labels(self):
        """Returns a list of the arc ids and labels.

        Returns:
            (:obj:`tuple(list)`): The arc ids and labels.
        """
        df = self.monitor_df
        fids = []
        flabels = []
        id_json = zip(df['id'].tolist(), df['json'].tolist())
        id_json = [(int(ii[0]), ii[1]) for ii in id_json if ii[1] and 'observation_point' not in ii[1]]
        for val in id_json:
            jd = json.loads(val[1].encode('utf8'))
            fids.append(val[0])
            label = jd.get('label', '')
            if not label:
                label = 'Monitor'
            flabels.append(label)
        return fids, flabels

    def get_point_comp_ids_labels(self, observation_point):
        """Returns a list of the point component ids and labels.

        Args:
            observation_point (:obj:`bool`): True if the desired list is for observation points.

        Returns:
            (:obj:`tuple(list)`): The point ids and labels.
        """
        df = self.monitor_df
        default_label = 'Monitor' if not observation_point else 'Measured value'
        fids = []
        flabels = []
        id_json = zip(df['id'].tolist(), df['json'].tolist())
        id_json = [(int(ii[0]), ii[1]) for ii in id_json if 'observation_point' in ii[1]]
        for val in id_json:
            jd = json.loads(val[1].encode())
            if observation_point and not jd.get('observation_point', False):
                continue
            if not observation_point and jd.get('observation_point', False):
                continue

            fids.append(val[0])
            label = jd.get('label', '')
            if not label:
                label = default_label
            flabels.append(label)
        return fids, flabels
