"""BcDataExitH class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import pandas as pd
import param

# 3. Aquaveo modules
from xms.guipy.param.param_enabler import ParamEnabler  # noqa: I202

# 4. Local modules


class BcDataExitH(param.Parameterized):
    """Bc type exit-h data."""
    water_surface_elevation_option = param.ObjectSelector(
        label='Water surface (WSE) option',
        default='Constant',
        objects=['Constant', 'Time series', 'Rating curve'],
        doc='option for specifying the WSE',
        precedence=1,
    )

    constant_wse = param.Number(
        label='Constant wse',
        default=0.0,
        doc='constant flow',
        precedence=2,
    )
    constant_wse_units = param.ObjectSelector(
        label='',
        default='Feet',
        objects=['Feet', 'Meters'],
        doc='units for constant WSE',
        precedence=3,
    )
    time_series_wse = param.DataFrame(
        default=pd.DataFrame(columns=['hrs', 'm or ft']),
        doc='time series for WSE',
        precedence=5,
    )
    time_series_wse_units = param.ObjectSelector(
        label='',
        default='hrs -vs- feet',
        objects=['hrs -vs- feet', 'hrs -vs- meters'],
        doc='units for time series WSE',
        precedence=6,
    )

    rating_curve = param.DataFrame(
        default=pd.DataFrame(columns=['vol/sec', 'WSE']),
        doc='time series for rating curve',
        precedence=7,
    )
    rating_curve_units = param.ObjectSelector(
        label='',
        default='cfs -vs- feet',
        objects=['cfs -vs- feet', 'cms -vs- meters'],
        doc='units for the rating curve',
        precedence=8,
    )
    wse_manning_n = param.Action(
        doc='defines the constant WSE from a tool',
        label='Populate using Channel Calculator...',
        precedence=10,
    )
    manning_n_calculator_inputs = param.String(
        default='',
        doc='json formatted string holding the inputs to the mannings n calculator.',
        precedence=-11,
    )

    def __init__(self):
        """Initializes the class."""
        super().__init__()
        self.enabler = ParamEnabler(
            param_cls=self,
            enablers=['water_surface_elevation_option'],
            depends={
                'Constant': ['constant_wse', 'constant_wse_units', 'wse_manning_n'],
                'Time series': ['time_series_wse', 'time_series_wse_units'],
                'Rating curve': ['rating_curve', 'rating_curve_units', 'wse_manning_n'],
            },
        )

    @param.depends('water_surface_elevation_option', watch=True)
    def _update_water_surface_elevation_option(self):
        """Watches changes made to the water_surface_elevation_option object."""
        self.enabler.do_enabling(force_disable=False)
