"""Qt delegate for the XY series curve button widget."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QSize, Qt
from PySide2.QtGui import QFontMetrics
from PySide2.QtWidgets import QApplication, QStyle, QStyleOptionComboBox

# 3. Aquaveo modules
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate  # noqa
from xms.guipy.widgets.qx_editable_combo_box import QxEditableComboBox

# 4. Local modules


class QxXySeriesCbxDelegate(QxCbxDelegate):
    """Combo box that is editable and shows the list of XY series names."""
    def __init__(self, parent=None):
        """Initializes the class.

        Args:
            parent (Something derived from :obj:`QWidget`): The parent window.
        """
        super().__init__(parent)
        self._model_index = None

    def get_strings(self, index=None):
        """Get the available combobox options for an item.

        Args:
            index (:obj:`QModelIndex`): Index of the item to get options for

        Returns:
            (:obj:`list[str]`): The specified item's available combobox options
        """
        if index is None:
            return []
        strings = self.parent().xy_series_names.copy()
        col = index.column()
        row = index.row()
        constants = self.parent().constants
        strings.append(constants[col][row])
        return strings

    def createEditor(self, parent, option, index):  # noqa: N802
        """Creates and populates an editable combobox widget.

        Args:
            parent (:obj:`QWidget`): Qt parent of the new editor widget
            option (:obj:`QStyleOptionViewItem`): The style options
            index (:obj:`QModelIndex`): Index of the item being edited

        Returns:
            (:obj:`QxEditableComboBox`): An editable combobox widget
        """
        self._model_index = index
        strings = self.get_strings(index)
        # Only the last item is editable
        self.cb = QxEditableComboBox(parent=parent, strings=strings, edit_indices={len(strings) - 1})
        self.cb.currentIndexChanged.connect(self.on_index_changed)
        self.cb.corrector.text_corrected.connect(self.on_text_corrected)
        return self.cb

    def sizeHint(self, option, index):  # noqa: N802
        """Help keep the size adjusted for custom painted combobox.

        Args:
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.

        Returns:
            (:obj:`QSize`): An appropriate size hint
        """
        hint = super(QxCbxDelegate, self).sizeHint(option, index)
        fm = QFontMetrics(option.font)
        cb_opt = QStyleOptionComboBox()
        cb_opt.rect = option.rect
        cb_opt.state = option.state | QStyle.State_Enabled
        strings = self.get_strings(index)
        for opt in strings:
            hint = hint.expandedTo(
                QApplication.style().sizeFromContents(
                    QStyle.CT_ComboBox, cb_opt, QSize(fm.boundingRect(opt).width(), hint.height())
                )
            )
        return hint

    def setEditorData(self, editor, index):  # noqa: N802
        """Sets the data to be displayed and edited by the editor for an item in the model.

        Args:
            editor (:obj:`QWidget`): The editor widget
            index (:obj:`QModelIndex`): Index of the item being edited
        """
        current_text = index.data(Qt.EditRole)
        cb_index = editor.findText(current_text, Qt.MatchFixedString)  # Case insensitive
        if cb_index == self.cb.count() - 1:
            editor.setCurrentIndex(cb_index)
            self.commitData.emit(self.cb)
            self.closeEditor.emit(self.cb)
        else:
            if cb_index >= 0:
                editor.blockSignals(True)
                editor.setCurrentIndex(cb_index)
                editor.blockSignals(False)
                self.commitData.emit(self.cb)
                self.closeEditor.emit(self.cb)
            editor.showPopup()

    def setModelData(self, editor, model, index):  # noqa: N802
        """Gets data from the editor widget and stores it in the model.

        Args:
            editor (:obj:`QWidget`): The editor widget
            model (:obj:`QAbstractItemModel`): The model to update
            index (:obj:`QModelIndex`): Index of the item being edited
        """
        cb_index = editor.currentIndex()
        # If it is valid, adjust the combobox
        if cb_index >= 0:
            text = editor.currentText()
            if cb_index == self.cb.count() - 1:
                constants = self.parent().constants
                constants[index.column()][index.row()] = text
            model.setData(index, text, Qt.EditRole)
            self.state_changed.emit(index)

    def on_index_changed(self, index):
        """Slot to close the editor when the user selects an option.

        Args:
            index (:obj:`int`): Index of the currently selected option
        """
        if index < self.cb.count() - 1:
            self.commitData.emit(self.cb)
            self.closeEditor.emit(self.cb)
            self.cb.hidePopup()
            self.cb.setParent(None)
            self.cb = None

    def on_text_corrected(self):
        """Slot to update data in the parent model after editable text has been formatted."""
        self.setModelData(self.cb, self.parent().runs_model, self._model_index)
