"""Migrate old DMI boundary conditions coverage attributes."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
import pandas as pd

# 3. Aquaveo modules
from xms.core.filesystem import filesystem

# 4. Local modules
from xms.srh.data.par.bc_data_param import BcDataParam
from xms.srh.migrate import migrate_consts as mgc


def add_sets_to_arc_db_data(widget_map, set_map):
    """Add entries into the BC arc database data dict for each arc in every set pair.

    Format xms.components gives us database set data is not terribly useful.

    Args:
        widget_map (:obj:`dict`): Dict of arc widget values for a particular coverage that we want to fill with
            the set arc attributes.
        set_map (:obj:`dict`): Dict of set widget values for a particular coverage as provided by the xms.components
            migrate_runner.

    """
    for _, set_atts in set_map.items():
        set_type = set_atts['cbxLineType'][0][2]
        try:
            set_widget = mgc.STRUCT_SET_WIDGETS[set_type]
        except KeyError:
            # This is an invalid set, just move on. The old DMI databases commonly got filled with garbage data,
            # particularly the set table.
            continue
        set_info = set_atts[set_widget]
        set_roles = {0: {'id': -1, 'role': ''}, 1: {'id': -1, 'role': ''}}
        for item in set_info:
            if item[0] == 'object':
                set_roles[item[1]]['id'] = item[2]
            else:  # role
                set_roles[item[1]]['role'] = item[2]
        # Add role assignment information to set att dict in useful place and add to bcs for both arcs in the pair.
        for set_arc in set_roles.values():
            if 'upstream' in set_arc['role'] or 'inflow' in set_arc['role']:
                set_atts['up_arc'] = set_arc['id']
            else:  # downstream
                set_atts['down_arc'] = set_arc['id']
        widget_map[set_atts['up_arc']] = set_atts
        widget_map[set_atts['down_arc']] = set_atts


class MigrateBcAtts:
    """Convert attributes on BC coverage arcs/sets to component-based data."""
    def __init__(self, proj_dir, cov_arcs, widget_map, hy8_widget_map):
        """Construct the migrator.

        Args:
            proj_dir (:obj:`str`): Path to the saved project's directory.
            cov_arcs (:obj:`list`): List of the xms.data_objects.Parameters.Arc objects in the obstructions coverage.
            widget_map (:obj:`dict`): Dict of arc widget values for a particular coverage that we want to fill with
                the set arc attributes. Set arcs should have attributes filled in from set database dict.
            hy8_widget_map (:obj:`dict`): Dict of HY-8 executable widget values keyed by hidden simulation id.

        """
        self._proj_dir = proj_dir
        self._arcs = cov_arcs
        self._widget_map = widget_map
        self._hy8_widget_map = hy8_widget_map
        self.bcs = {}

    def migrate(self):
        """Migrate the BC arc/set attributes."""
        for arc in self._arcs:
            arc_id = arc.id
            if arc_id in self.bcs:
                continue  # This arc is part of a set that we already processed

            if arc_id not in self._widget_map:  # Default atts (wall)
                self.bcs[arc_id] = BcDataParam()
                continue

            bc_db_atts = self._widget_map[arc_id]
            bc_param = BcDataParam()
            bc_param.bc_type = mgc.BC_TYPE_DB_TO_COMP[bc_db_atts['cbxLineType'][0][2]]
            if bc_param.bc_type in mgc.STRUCT_SET_WIDGETS:
                # If part of a set, use same BcDataParam for both arcs. Will get assigned different comp ids.
                bc_param.arcs.arc_id_0 = bc_db_atts['up_arc']
                bc_param.arcs.arc_option_0 = 'Upstream'
                bc_param.arcs.arc_id_1 = bc_db_atts['down_arc']
                bc_param.arcs.arc_option_1 = 'Downstream'
                other_arc = bc_param.arcs.arc_id_0 if bc_param.arcs.arc_id_0 != arc_id else bc_param.arcs.arc_id_1
                self.bcs[other_arc] = bc_param

            self.bcs[arc_id] = bc_param
            self._fill_bc_atts_from_db(bc_db_atts, bc_param)

    def _fill_bc_atts_from_db(self, bc_db_atts, bc_param):
        """Fill a BcDataParam object from values in the database.

        Args:
            bc_db_atts (:obj:`dict`): Dict of database widget values for the BC arc in the format passed in by
                xms.components run_migrate
            bc_param (:obj:`BcDataParam`): The BC data to fill

        """
        self._migrate_wall_atts(bc_db_atts, bc_param)
        self._migrate_inlet_q_and_exit_q_atts(bc_db_atts, bc_param)
        self._migrate_exit_h_atts(bc_db_atts, bc_param)
        self._migrate_inlet_sc_atts(bc_db_atts, bc_param)
        self._migrate_culvert_atts(bc_db_atts, bc_param)
        self._migrate_hy8_atts(bc_db_atts, bc_param)
        self._migrate_weir_atts(bc_db_atts, bc_param)
        self._migrate_gate_atts(bc_db_atts, bc_param)
        self._migrate_pressure_atts(bc_db_atts, bc_param)
        self._migrate_internal_sink_atts(bc_db_atts, bc_param)
        self._migrate_internal_link_atts(bc_db_atts, bc_param)
        self._migrate_sediment_inflow_atts(bc_db_atts, bc_param)

    @staticmethod
    def _migrate_wall_atts(bc_db_atts, bc_param):
        """Fill wall BC attributes from values in the database.

        Args:
            bc_db_atts (:obj:`dict`): Dict of database widget values for the BC arc in the format passed in by
                xms.components run_migrate
            bc_param (:obj:`BcDataParam`): The BC data to fill

        """
        bc_param.wall.extra_wall_roughness = bc_db_atts['cbxWallRoughOnOff'][0][2] == 'On'
        bc_param.wall.roughness = bc_db_atts['edtWallRoughness'][0][2]

    @staticmethod
    def _migrate_inlet_q_and_exit_q_atts(bc_db_atts, bc_param):
        """Fill Inlet-Q and Exit-Q BC attributes from values in the database.

        Args:
            bc_db_atts (:obj:`dict`): Dict of database widget values for the BC arc in the format passed in by
                xms.components run_migrate
            bc_param (:obj:`BcDataParam`): The BC data to fill

        """
        discharge_opt = (
            bc_db_atts['cbxDischarge'][0][2] if bc_db_atts['cbxDischarge'][0][2] == 'Constant' else 'Time series'
        )
        bc_param.inlet_q.discharge_option = discharge_opt
        bc_param.exit_q.discharge_option = discharge_opt
        constant_discharge = bc_db_atts['floConstDischarge'][0][2]
        constant_units = bc_db_atts['floConstDischarge'][1][2]
        bc_param.inlet_q.constant_q = constant_discharge
        bc_param.exit_q.constant_q = constant_discharge
        bc_param.inlet_q.constant_q_units = constant_units
        bc_param.exit_q.constant_q_units = constant_units
        if 'crvTsDischarge' in bc_db_atts:
            discharge_x = bc_db_atts['crvTsDischarge'][0][2][2]
            discharge_y = bc_db_atts['crvTsDischarge'][0][2][3]
            if discharge_x:
                bc_param.inlet_q.time_series_q = pd.DataFrame({'hrs': discharge_x, 'vol_per_sec': discharge_y})
                bc_param.exit_q.time_series_q = pd.DataFrame({'hrs': discharge_x, 'vol/sec': discharge_y})
            bc_param.inlet_q.time_series_q_units = bc_db_atts['cbxTimeSeriesUnits'][0][2]
            bc_param.exit_q.time_series_q_units = bc_db_atts['cbxTimeSeriesUnits'][0][2]
        bc_param.inlet_q.distribution_at_inlet = bc_db_atts['cbxInletDistrib'][0][2]

    @staticmethod
    def _migrate_exit_h_atts(bc_db_atts, bc_param):
        """Fill Exit-H BC attributes from values in the database.

        Args:
            bc_db_atts (:obj:`dict`): Dict of database widget values for the BC arc in the format passed in by
                xms.components run_migrate
            bc_param (:obj:`BcDataParam`): The BC data to fill

        """
        bc_param.exit_h.water_surface_elevation_option = mgc.CURVE_OPTION_DB_TO_COMP[bc_db_atts['cbxWSE'][0][2]]
        bc_param.exit_h.constant_wse = bc_db_atts['wseElevation'][0][2]
        bc_param.exit_h.constant_wse_units = bc_db_atts['wseElevation'][1][2]
        if 'crvTsWse' in bc_db_atts:
            wse_x = bc_db_atts['crvTsWse'][0][2][2]
            wse_y = bc_db_atts['crvTsWse'][0][2][3]
            if wse_x:
                bc_param.exit_h.time_series_wse = pd.DataFrame({'hrs': wse_x, 'm or ft': wse_y})
            bc_param.exit_h.time_series_wse_units = bc_db_atts['cbxWSEFileUnits'][0][2]

        if 'crvRatingWse' in bc_db_atts:
            wse_rc_x = bc_db_atts['crvRatingWse'][0][2][2]
            wse_rc_y = bc_db_atts['crvRatingWse'][0][2][3]
            if wse_rc_x:
                bc_param.exit_h.rating_curve = pd.DataFrame({'vol/sec': wse_rc_x, 'WSE': wse_rc_y})
            bc_param.exit_h.rating_curve_units = bc_db_atts['cbxQWSEFileUnits'][0][2]

    @staticmethod
    def _migrate_inlet_sc_atts(bc_db_atts, bc_param):
        """Fill Inlet-SC BC attributes from values in the database.

        Args:
            bc_db_atts (:obj:`dict`): Dict of database widget values for the BC arc in the format passed in by
                xms.components run_migrate
            bc_param (:obj:`BcDataParam`): The BC data to fill

        """
        bc_param.inlet_sc.distribution_at_inlet = bc_db_atts['cbxInletDistrib'][0][2]
        bc_param.inlet_sc.discharge_q_option = mgc.CURVE_OPTION_DB_TO_COMP[bc_db_atts['cbxSCDischarge'][0][2]]
        bc_param.inlet_sc.discharge_q_units = (
            'English' if bc_db_atts['cbxSCUnits'][0][2] == 'U.S. Customary' else 'Metric'
        )
        bc_param.inlet_sc.constant_q = bc_db_atts['scDischarge'][0][2]
        bc_param.inlet_sc.constant_wse = bc_db_atts['scinElevation'][0][2]
        time_series_q_x = None
        time_series_q_y = None
        if 'crvSCTsDischarge' in bc_db_atts:  # Widget name changed from 3-4
            time_series_q_x = bc_db_atts['crvSCTsDischarge'][0][2][2]
            time_series_q_y = bc_db_atts['crvSCTsDischarge'][0][2][3]
        elif 'crvSCTvsQ' in bc_db_atts:
            time_series_q_x = bc_db_atts['crvSCTvsQ'][0][2][2]
            time_series_q_y = bc_db_atts['crvSCTvsQ'][0][2][3]
        if time_series_q_x:
            bc_param.inlet_sc.time_series_q = pd.DataFrame({'hrs': time_series_q_x, 'vol_per_sec': time_series_q_y})
        sc_wse = bc_param.inlet_sc.water_surface_elevation
        sc_wse.water_elevation_option = mgc.CURVE_OPTION_DB_TO_COMP[bc_db_atts['cbxSCWSE'][0][2]]
        if 'crvSCTsWse' in bc_db_atts:
            time_series_wse_x = bc_db_atts['crvSCTsWse'][0][2][2]
            time_series_wse_y = bc_db_atts['crvSCTsWse'][0][2][3]
            if time_series_wse_x:
                sc_wse.time_series_wse = pd.DataFrame({'hrs': time_series_wse_x, 'elev': time_series_wse_y})
        rating_crv_wse_x = None
        rating_crv_wse_y = None
        if 'crvSCRatingWse' in bc_db_atts:  # Widget name changed from 3-4
            rating_crv_wse_x = bc_db_atts['crvSCRatingWse'][0][2][2]
            rating_crv_wse_y = bc_db_atts['crvSCRatingWse'][0][2][3]
        elif 'crvSCQvsZ' in bc_db_atts:
            rating_crv_wse_x = bc_db_atts['crvSCQvsZ'][0][2][2]
            rating_crv_wse_y = bc_db_atts['crvSCQvsZ'][0][2][3]
        if rating_crv_wse_x:
            sc_wse.rating_curve = pd.DataFrame({'vol_per_sec': rating_crv_wse_x, 'WSE': rating_crv_wse_y})

    @staticmethod
    def _migrate_culvert_atts(bc_db_atts, bc_param):
        """Fill Culvert BC attributes from values in the database.

        Args:
            bc_db_atts (:obj:`dict`): Dict of database widget values for the BC arc in the format passed in by
                xms.components run_migrate
            bc_param (:obj:`BcDataParam`): The BC data to fill

        """
        if 'edtCulvertInvertElevationUp' in bc_db_atts:
            bc_param.culvert.invert_elevation = bc_db_atts['edtCulvertInvertElevationUp'][0][2]
            bc_param.culvert.barrel_height = bc_db_atts['edtCulvertBarrelHeight'][0][2]
            bc_param.culvert.barrel_length = bc_db_atts['edtCulvertBarrelLength'][0][2]
            bc_param.culvert.barrel_area = bc_db_atts['edtCulvertBarrelArea'][0][2]
            bc_param.culvert.barrel_hydraulic_radius = bc_db_atts['edtCulvertBarrelRh'][0][2]
            bc_param.culvert.barrel_slope = bc_db_atts['edtCulvertBarrelSlope'][0][2]
            bc_param.culvert.units = 'Feet' if bc_db_atts['cbxCulvertUnits'][0][2] == 'ft' else 'Meters'
            bc_param.culvert.num_barrels = bc_db_atts['edtCulvertBarrelNumber'][0][2]
            bc_param.culvert.entrance_type = bc_db_atts['cbxCulvertEntranceType'][0][2]
            if 'cbxCulvertInletType' in bc_db_atts:
                bc_param.culvert.inlet_coefficients = bc_db_atts['cbxCulvertInletType'][0][2]
            bc_param.culvert.loss_coefficient = bc_db_atts['edtCulvertInletKe'][0][2]
            bc_param.culvert.mannings_n = bc_db_atts['edtCulvertN'][0][2]
            if 'togCulvertTotalHead' in bc_db_atts:
                bc_param.culvert.total_head = bc_db_atts['togCulvertTotalHead'][0][2] == 1
            bc_param.culvert.crest_elevation = bc_db_atts['edtCulvertWeirCrestElevation'][0][2]
            bc_param.culvert.length_of_weir_over_culvert = bc_db_atts['edtCulvertWeirLength'][0][2]
            bc_param.culvert.weir.type = bc_db_atts['cbxCulvertWeirType'][0][2].title()
            bc_param.culvert.weir.cw = bc_db_atts['edtCulvertWeirCW'][0][2]
            bc_param.culvert.weir.a = bc_db_atts['edtCulvertWeirA'][0][2]
            bc_param.culvert.weir.b = bc_db_atts['edtCulvertWeirB'][0][2]

    @staticmethod
    def _migrate_weir_atts(bc_db_atts, bc_param):
        """Fill Weir BC attributes from values in the database.

        Args:
            bc_db_atts (:obj:`dict`): Dict of database widget values for the BC arc in the format passed in by
                xms.components run_migrate
            bc_param (:obj:`BcDataParam`): The BC data to fill

        """
        bc_param.weir.units = 'Feet' if bc_db_atts['cbxWeirUnits'][0][2] == 'ft' else 'Meters'
        bc_param.weir.crest_elevation = bc_db_atts['edtWeirCrestElevation'][0][2]
        bc_param.weir.length = bc_db_atts['edtWeirLength'][0][2]
        bc_param.weir.type.type = bc_db_atts['cbxWeirType'][0][2].title()
        bc_param.weir.type.cw = bc_db_atts['edtWeirCW'][0][2]
        bc_param.weir.type.a = bc_db_atts['edtWeirA'][0][2]
        bc_param.weir.type.b = bc_db_atts['edtWeirB'][0][2]

    @staticmethod
    def _migrate_gate_atts(bc_db_atts, bc_param):
        """Fill Gate BC attributes from values in the database.

        Args:
            bc_db_atts (:obj:`dict`): Dict of database widget values for the BC arc in the format passed in by
                xms.components run_migrate
            bc_param (:obj:`BcDataParam`): The BC data to fill

        """
        bc_param.gate.units = 'Feet' if bc_db_atts['cbxGateUnits'][0][2] == 'ft' else 'Meters'
        bc_param.gate.crest_elevation = bc_db_atts['edtGateCrestElevation'][0][2]
        bc_param.gate.height = bc_db_atts['edtGateHeight'][0][2]
        bc_param.gate.width = bc_db_atts['edtGateWidth'][0][2]
        bc_param.gate.contract_coefficient = bc_db_atts['edtGateOrifice'][0][2]
        bc_param.gate.type.type = bc_db_atts['cbxGateType'][0][2].title()
        bc_param.gate.type.cw = bc_db_atts['edtGateCW'][0][2]
        bc_param.gate.type.a = bc_db_atts['edtGateA'][0][2]
        bc_param.gate.type.b = bc_db_atts['edtGateB'][0][2]

    @staticmethod
    def _migrate_pressure_atts(bc_db_atts, bc_param):
        """Fill Pressure BC attributes from values in the database.

        Args:
            bc_db_atts (:obj:`dict`): Dict of database widget values for the BC arc in the format passed in by
                xms.components run_migrate
            bc_param (:obj:`BcDataParam`): The BC data to fill

        """
        bc_param.pressure.units = 'Feet' if bc_db_atts['cbxPressureUnits'][0][2] == 'ft' else 'Meters'
        bc_param.pressure.upstream_elevation = bc_db_atts['edtPressureElevationUp'][0][2]
        bc_param.pressure.downstream_elevation = bc_db_atts['edtPressureElevationDown'][0][2]
        bc_param.pressure.roughness = bc_db_atts['edtPressureCoef'][0][2]
        if 'cbxPressureType' in bc_db_atts:
            bc_param.pressure.ceiling_type = bc_db_atts['cbxPressureType'][0][2]
        if 'togPressureOvertop' in bc_db_atts:
            bc_param.pressure.overtopping = bc_db_atts['togPressureOvertop'][0][2] == 1
            bc_param.pressure.weir_crest_elevation = bc_db_atts['edtPressWeirCrestElevation'][0][2]
            bc_param.pressure.weir_length = bc_db_atts['edtPressWeirLength'][0][2]
            bc_param.pressure.weir_type.type = bc_db_atts['cbxPressWeirType'][0][2].title()
            bc_param.pressure.weir_type.cw = bc_db_atts['edtPressWeirCW'][0][2]
            bc_param.pressure.weir_type.a = bc_db_atts['edtPressWeirA'][0][2]
            bc_param.pressure.weir_type.b = bc_db_atts['edtPressWeirB'][0][2]

    @staticmethod
    def _migrate_internal_sink_atts(bc_db_atts, bc_param):
        """Fill Internal Sink BC attributes from values in the database.

        Args:
            bc_db_atts (:obj:`dict`): Dict of database widget values for the BC arc in the format passed in by
                xms.components run_migrate
            bc_param (:obj:`BcDataParam`): The BC data to fill

        """
        if 'cbxInternalQType' in bc_db_atts:
            bc_param.internal_sink.sink_flow_type = mgc.CURVE_OPTION_DB_TO_COMP[bc_db_atts['cbxInternalQType'][0][2]]
            bc_param.internal_sink.constant_q = bc_db_atts['floConstIntDischarge'][0][2]
            bc_param.internal_sink.constant_q_units = bc_db_atts['floConstIntDischarge'][1][2]
            if 'crvTsIntDischarge' in bc_db_atts:
                internal_time_series_q_x = bc_db_atts['crvTsIntDischarge'][0][2][2]
                internal_time_series_q_y = bc_db_atts['crvTsIntDischarge'][0][2][3]
                if internal_time_series_q_x:
                    bc_param.internal_sink.time_series_q = pd.DataFrame(
                        {
                            'hrs': internal_time_series_q_x,
                            'vol_per_sec': internal_time_series_q_y
                        }
                    )
                bc_param.internal_sink.time_series_q_units = bc_db_atts['cbxTSIntUnits'][0][2]
            bc_param.internal_sink.weir_coeff = bc_db_atts['edtIntWeirCw'][0][2]
            bc_param.internal_sink.weir_crest_elevation = bc_db_atts['edtIntWeirZb'][0][2]
            bc_param.internal_sink.weir_length_across = bc_db_atts['edtIntWeirLw'][0][2]
            bc_param.internal_sink.weir_units = bc_db_atts['cbxIntWeirUnits'][0][2].title()
            bc_param.internal_sink.weir_use_total_head = bc_db_atts['togIntWeirTHead'][0][2] == 1
            if 'crvIntQvsZ' in bc_db_atts:
                internal_rating_curve_x = bc_db_atts['crvIntQvsZ'][0][2][2]
                internal_rating_curve_y = bc_db_atts['crvIntQvsZ'][0][2][3]
                if internal_rating_curve_x:
                    bc_param.internal_sink.rating_curve = pd.DataFrame(
                        {
                            'vol_per_sec': internal_rating_curve_x,
                            'WSE': internal_rating_curve_y
                        }
                    )
                bc_param.internal_sink.rating_curve_units = bc_db_atts['cbxRCIntUnits'][0][2]

    @staticmethod
    def _migrate_internal_link_atts(bc_db_atts, bc_param):
        """Fill Internal Sink BC attributes from values in the database.

        Args:
            bc_db_atts (:obj:`dict`): Dict of database widget values for the BC arc in the format passed in by
                xms.components run_migrate
            bc_param (:obj:`BcDataParam`): The BC data to fill

        """
        if 'cbxLinkQType' in bc_db_atts:
            bc_param.link.inflow_type = mgc.CURVE_OPTION_DB_TO_COMP[bc_db_atts['cbxLinkQType'][0][2]]
            bc_param.link.constant_q = bc_db_atts['floConstLinkDischarge'][0][2]
            bc_param.link.constant_q_units = bc_db_atts['floConstLinkDischarge'][1][2]
            if 'crvTsLinkDischarge' in bc_db_atts:
                link_time_series_q_x = bc_db_atts['crvTsLinkDischarge'][0][2][2]
                link_time_series_q_y = bc_db_atts['crvTsLinkDischarge'][0][2][3]
                if link_time_series_q_x:
                    bc_param.link.time_series_q = pd.DataFrame(
                        {
                            'hrs': link_time_series_q_x,
                            'vol_per_sec': link_time_series_q_y
                        }
                    )
                bc_param.link.time_series_q_units = bc_db_atts['cbxTSLinkUnits'][0][2]
            # bc_param.link.weir.coeff = bc_db_atts['edtLinkWeirCw'][0][2]
            bc_param.link.weir.type.cw = bc_db_atts['edtLinkWeirCw'][0][2]
            bc_param.link.weir.type.a = 16.4
            bc_param.link.weir.type.b = 0.432
            bc_param.link.weir.type.type = 'User'

            bc_param.link.weir.crest_elevation = bc_db_atts['edtLinkWeirZb'][0][2]
            bc_param.link.weir.length = bc_db_atts['edtLinkWeirLw'][0][2]
            bc_param.link.weir.units = bc_db_atts['cbxLinkWeirUnits'][0][2].title()
            bc_param.link.weir.total_head = bc_db_atts['togLinkWeirTHead'][0][2] == 1
            if 'crvLnkQvsZ' in bc_db_atts:
                link_rating_curve_x = bc_db_atts['crvLnkQvsZ'][0][2][2]
                link_rating_curve_y = bc_db_atts['crvLnkQvsZ'][0][2][3]
                if link_rating_curve_x:
                    bc_param.link.rating_curve = pd.DataFrame(
                        {
                            'vol_per_sec': link_rating_curve_x,
                            'WSE': link_rating_curve_y
                        }
                    )
                bc_param.link.rating_curve_units = bc_db_atts['cbxRCLinkUnits'][0][2]
            bc_param.link.link_lag_method = bc_db_atts['cbxLinkLagMethod'][0][2]
            bc_param.link.specified_lag = bc_db_atts['edtLinkLag'][0][2]
            bc_param.link.conduit_length = bc_db_atts['edtLinkLc'][0][2]
            bc_param.link.conduit_diameter = bc_db_atts['edtLinkDc'][0][2]
            bc_param.link.conduit_slope = bc_db_atts['edtLinkS'][0][2]
            bc_param.link.conduit_mannings = bc_db_atts['edtLinkN'][0][2]
            bc_param.link.conduit_units = bc_db_atts['cbxLinkLagUnits'][0][2].title()

    def _migrate_hy8_atts(self, bc_db_atts, bc_param):
        """Fill HY-8 culvert BC attributes from values in the database.

        Args:
            bc_db_atts (:obj:`dict`): Dict of database widget values for the BC arc in the format passed in by
                xms.components run_migrate
            bc_param (:obj:`BcDataParam`): The BC data to fill

        """
        bc_param.hy8_culvert.units = 'English' if bc_db_atts['cbxHY8Units'][0][2] == 'U.S. Customary' else 'Metric'
        hy8_exe_sim_id = bc_db_atts['exeHY8'][0][2]
        if hy8_exe_sim_id in self._hy8_widget_map:
            bc_param.hy8_culvert.hy8_crossing_guid = self._hy8_widget_map[hy8_exe_sim_id]['edtHY8Guid'][0][2]
        if 'togHY8TotalHead' in bc_db_atts:
            bc_param.hy8_culvert.total_head = bc_db_atts['togHY8TotalHead'][0][2] == 1

    def _migrate_sediment_inflow_atts(self, bc_db_atts, bc_param):
        """Fill Sediment Inflow BC attributes from values in the database.

        Args:
            bc_db_atts (:obj:`dict`): Dict of database widget values for the BC arc in the format passed in by
                xms.components run_migrate
            bc_param (:obj:`BcDataParam`): The BC data to fill

        """
        if 'cbxSedDischarge' in bc_db_atts:
            bc_param.sediment_inflow.sediment_discharge_type = bc_db_atts['cbxSedDischarge'][0][2]
            sediment_file = bc_db_atts['edtSedDischargeFile'][0][2]
            if sediment_file and sediment_file != '(none selected)':
                # Might be stored as relative from the project database file, depending on version.
                sed_file = filesystem.resolve_relative_path(self._proj_dir, sediment_file)
                if os.path.isfile(sed_file):
                    bc_param.sediment_inflow.sediment_file = sed_file
