"""The executable script used by SMS for running SRH-Pre with PEST models."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import os

# 2. Third party modules
import orjson

# 3. Aquaveo modules
from xms.core.filesystem import filesystem

# 4. Local modules


def get_computed_values(model_data, logger=None, folder=None):
    """Reads the SRH2D Monitor point output files and returns the computed values.

    Args:
        model_data (:obj:`dict`): Dict with strings as the keys and various lists and strings as the values containing
            the model data
        logger (:obj:`logger`): A logger for printing output
        folder (:obj:`str`): The full-path folder name or a folder relative to the current working folder
    """
    case_name = model_data['CASE_NAME']
    num_monitor = len(model_data['MONITOR_PTS'])
    if folder:
        monitor_file_base = os.path.join(folder, 'Output_MISC', case_name)
    else:
        monitor_file_base = os.path.join('Output_MISC', case_name)
    monitor_files = [f'{monitor_file_base}_PT{i}.dat' for i in range(1, num_monitor + 1)]
    values = []
    if logger:
        logger.info(f'{model_data}')
        logger.info(f'{monitor_files}')
    for file_index, filename in enumerate(monitor_files):
        if logger:
            logger.info(f"Use Monitor point index {file_index}: {model_data['MONITOR_PTS'][file_index]}")
        if model_data['MONITOR_PTS'][file_index] is False:
            if logger:
                logger.info(f'skipped monitor point index {file_index}')
            continue
        if logger:
            logger.info(f'monitor point index {file_index}')
        value = 0.0
        if os.path.exists(filename):
            with open(filename) as f:
                lines = f.readlines()
                f.close()
            if lines:
                first_line = lines[0].split()
                index_wse = 0
                index_elev = 0
                for idx, card in enumerate(first_line):
                    if card.startswith('Water_Elev'):
                        index_wse = idx
                    elif card.startswith('Bed_Elev'):
                        index_elev = idx
                for line in reversed([x.strip() for x in lines]):
                    if line and not line.isspace():
                        line_data = line.split()
                        value = float(line_data[index_wse])
                        if value < -998:
                            value = float(line_data[index_elev])
                        break
            if logger:
                logger.info(f'monitor point value: {value}')
        values.append(value)
    return values


def run_model(json_file):
    """Call SRH-Pre and SRH via command line scripts using the specified run options for an SRH-2D PEST model.

    Args:
        json_file (:obj:`str`): Json file containing the options for running SRH-Pre and SRH on the PEST model
    """
    if json_file:
        with open(json_file.strip(), 'rb') as f:
            data = orjson.loads(f.read())
            f.close()
        file_base = data['PROJECT_NAME']
        os.system('start /wait /min cmd /c pest_run_pre.bat')
        os.system('start /wait /min cmd /c pest_run_srh.bat')
        with open(f'{file_base}_obs_out.txt', 'w') as of:
            filesystem.removefile('srh_pest_model.log')
            logger = logging.getLogger('srh_pest_model')
            handler = logging.FileHandler('srh_pest_model.log')
            logger.addHandler(handler)
            logger.setLevel(logging.INFO)
            values = get_computed_values(data, logger)
            for value in values:
                of.write(f'{value}\n')
            of.close()
