"""This is a widget for case data table."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QItemSelectionModel, QModelIndex, Qt

# 3. Aquaveo modules
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.models.rename_model import RenameModel
from xms.guipy.widgets.basic_table_widget import BasicTableWidget

# 4. Local modules
from xms.stwave.data import stwave_consts as const


class CaseDataTable(BasicTableWidget):
    """Class for a table to hold case data information."""
    def __init__(self, reftime, case_data, parent):
        """Constructor.  Initializes and sets up the Case data table.

        Args:
            reftime (:obj:'datetime`): The global time.
            case_data (:obj:`pandas.Dataframe`): The case data to display in the table.
            parent (Something derived from :obj:`QWidget`): The parent window.
        """
        super().__init__(parent)
        self.select_col = 0
        self.rename_model = RenameFilterModel(column_names=case_data.columns, dialog_parent=parent, reftime=reftime,
                                              parent=self)
        self.model = QxPandasTableModel(data_frame=case_data)
        self.model.set_default_values({column_name: 0.0 for column_name in case_data.columns})
        self._setup_ui({},
                       False, False, filter_model=self.rename_model)

    def on_btn_add_row(self):
        """Called when a new row is added to the table."""
        row_idx = self.model.rowCount()
        self.model.insertRows(row_idx, 1)
        new_index = self.model.index(row_idx, 0)
        self.table_view.selectionModel().setCurrentIndex(
            new_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
        )

    def on_btn_delete_row(self):
        """Called when a row is removed from the table."""
        indices = self.table_view.selectionModel().selectedIndexes()
        next_select_row = -1
        delete_rows = {index.row() if index.isValid() else -1 for index in indices}
        delete_count = 0
        for index in delete_rows:
            if index < 0:
                continue
            delete_row = index - delete_count
            next_select_row = delete_row - 1
            self.model.removeRows(delete_row, 1)
            delete_count += 1
        if next_select_row >= 0:
            select_index = self.table_view.model().index(next_select_row, self.select_col)
            self.table_view.selectionModel().setCurrentIndex(
                select_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
            )


class RenameFilterModel(RenameModel):
    """Derived class from RenameModel."""
    def __init__(self, column_names, dialog_parent, reftime, parent=None):
        """Constructor.  Used in the Case data table for handling columns based on other STWAVE settings.

        Args:
            column_names (list): The column names.
            dialog_parent (:obj:`ModelControlDialog`): The parent STWAVE Model Control dialog.
            reftime (:obj:`datetime`): The global time.
            parent (Something derived from :obj:`QObject`): The parent object.
        """
        super().__init__(column_names, parent)
        self._dialog_parent = dialog_parent
        self.ref_time = reftime

    def filterAcceptsColumn(self, source_column: int, source_parent: QModelIndex) -> bool:  # noqa: N802
        """Override for filterAcceptsColumn.

        The Wind direction and Wind magnitude columns depend on the Source terms and Wind combo boxes, while the
        Water level column depends on the Surge combo box.

        Args:
            source_column (int): The source column.
            source_parent (:obj:`PySide2.QtCore.QModelIndex`) The source model index.

        Returns:
            (bool): True if the column is to be used, based on other STWAVE settings.
        """
        if source_column in [1, 2]:
            # Wind direction and Wind magnitude columns
            source = self._dialog_parent.ui.source_terms_combo
            wind = self._dialog_parent.ui.wind_combo
            if source.currentText() == const.SOURCE_PROP_AND_TERMS and wind.currentText() == const.OPT_CONST:
                return True
        elif source_column == 3:
            # Water level column
            surge = self._dialog_parent.ui.surge_combo
            if surge.currentText() == const.OPT_CONST:
                return True
        elif source_column == 0:
            # Time column
            return True
        return False

    def headerData(self, section, orientation, role=Qt.DisplayRole):  # noqa: N802
        """Override for headerData."""
        if role != Qt.DisplayRole:
            return super().headerData(section, orientation, role)

        if orientation == Qt.Horizontal:
            count = self.columnCount()
            if count == 2 and section == 1:
                return self.column_names[3]
            return self.column_names[section]
        else:
            return section + 1  # 1-base row number for vertical header
