"""A script for tracking the progress an STWAVE simulation run."""

# 1. Standard Python modules
import datetime
import re

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query

# 4. Local modules
from xms.stwave.data.simulation_data import SimulationData
from xms.stwave.file_io.dataset_writer import get_time_string


class ProgressTracker:
    """A class for tracking the progress of an STWAVE simulation run."""
    prog = None
    query = None
    echo_file = None
    echo_pos = 0
    timesteps = []
    current_ts = 0

    def __init__(self):
        """Constructor that does nothing."""
        pass

    @staticmethod
    def progress_function():
        """Progress is calculated and sent to query as a percentage."""
        if not ProgressTracker.echo_file:
            ProgressTracker.echo_file = ProgressTracker.prog.command_line_output_file

        found_iter = False
        try:
            with open(ProgressTracker.echo_file, "r") as f:
                f.seek(ProgressTracker.echo_pos)
                echo_line = f.readline()
                while echo_line:
                    if echo_line.endswith('\n') or echo_line.endswith('\r'):
                        result = re.match(r'Wave\s*IDD\s*=\s*([0-9]+)', echo_line.strip(), re.IGNORECASE)
                        if result:
                            ProgressTracker.current_ts += 1
                            found_iter = True
                            ProgressTracker.echo_pos = f.tell()
                    echo_line = f.readline()
        except Exception:
            pass  # File might not exist yet

        if found_iter:
            percent_done = (ProgressTracker.current_ts / len(ProgressTracker.timesteps)) * 100.0
            ProgressTracker.query.update_progress_percent(percent_done)

    @staticmethod
    def start_tracking():
        """Entry point for the STWAVE progress script."""
        ProgressTracker.query = Query(progress_script=True)
        session = ProgressTracker.query.xms_agent.session

        # Get the simulation's hidden component
        sim_uuid = ProgressTracker.query.current_item_uuid()  # Get the simulation uuid
        sim_comp = ProgressTracker.query.item_with_uuid(sim_uuid, model_name='STWAVE', unique_name='Sim_Component')
        data = SimulationData(sim_comp.main_file)

        # get the snap case times
        gt = ProgressTracker.query.global_time
        ProgressTracker.timesteps = [get_time_string(gt + datetime.timedelta(seconds=float(x)))
                                     for x in data.case_times['Time']]
        ProgressTracker.num_timesteps = len(data.case_times['Time'])
        ProgressTracker.prog = session.progress_loop
        ProgressTracker.prog.set_progress_function(ProgressTracker.progress_function)
        ProgressTracker.prog.start_loop()
