"""PolygonsFromArcsTool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.tool_core import IoDirection, Tool

# 4. Local modules
from xms.tool.algorithms.coverage.polygons_from_arcs import polygons_from_arcs

ARG_INPUT_COVERAGE = 0
ARG_ELEMENT_WIDTH = 1
ARG_NUMBER_ELEMENTS = 2
ARG_BIAS = 3
ARG_PINCH_ENDS = 4
ARG_SET_MIN_SEG_LENGTH = 5
ARG_MIN_SEG_LENGTH = 6
ARG_OUTPUT_COVERAGE = 7


class PolygonsFromArcsTool(Tool):
    """Tool to convert arcs to polygons using a buffer max_dist."""

    def __init__(self):
        """Initializes the class."""
        super().__init__(name='Polygons from Arcs')

    def initial_arguments(self):
        """Get initial arguments for tool.

        Must override.

        Returns:
            (list): A list of the initial tool arguments.
        """
        arguments = [
            self.coverage_argument(name='input_coverage', description='Input coverage'),
            self.float_argument(name='element_width', description='Average element/cell width', value=0.0),
            self.integer_argument(name='number_elements', description='Number of elements/cells (must be 1 or even)',
                                  value=2, min_value=1),
            self.float_argument(name='bias', description='Bias (0.01-100.0)', value=1.0, min_value=0.01,
                                max_value=100.0),
            self.bool_argument(name='pinch_ends', description='Pinch free ends', value=True, hide=True),
            self.bool_argument(name='set_min_seg_length', description='Set minimum segment length', value=False),
            self.float_argument(name='min_seg_length', description='Minimum segment length', value=1.0, optional=True),
            self.coverage_argument(name='output_coverage', description='Output coverage',
                                   io_direction=IoDirection.OUTPUT)
        ]
        return arguments

    def enable_arguments(self, arguments):
        """Called to show/hide arguments, change argument values and add new arguments.

        Args:
            arguments(list): The tool arguments.
        """
        arguments[ARG_MIN_SEG_LENGTH].show = arguments[ARG_SET_MIN_SEG_LENGTH].value

    def validate_arguments(self, arguments):
        """Called to determine if arguments are valid.

        Args:
            arguments (list): The tool arguments.

        Returns:
            (dict): Dictionary of errors for arguments.
        """
        errors = {}
        # Validate the element/cell width
        if arguments[ARG_ELEMENT_WIDTH].value == 0.0:
            errors[arguments[ARG_ELEMENT_WIDTH].name] = 'Average element/cell width must not be 0.0.'
        if arguments[ARG_NUMBER_ELEMENTS].value % 2 != 0 and arguments[ARG_NUMBER_ELEMENTS].value != 1:
            errors[arguments[ARG_NUMBER_ELEMENTS].name] = 'Number of elements must be 1 or an even number.'
        return errors

    def run(self, args):
        """Override to run the tool.

        Args:
            args (list): The tool arguments.
        """
        input_coverage = self.get_input_coverage(args[ARG_INPUT_COVERAGE].text_value)
        min_length = args[ARG_MIN_SEG_LENGTH].value if args[ARG_SET_MIN_SEG_LENGTH].value is True else None

        try:
            output_cov = polygons_from_arcs(input_coverage, args[ARG_ELEMENT_WIDTH].value,
                                            args[ARG_NUMBER_ELEMENTS].value, args[ARG_BIAS].value,
                                            args[ARG_OUTPUT_COVERAGE].value, self.logger,
                                            args[ARG_PINCH_ENDS].value, min_length,
                                            self.default_wkt)
        except RuntimeError as e:
            self.fail(str(e))
        else:
            self.set_output_coverage(output_cov, args[ARG_OUTPUT_COVERAGE])
