"""DatasetArgument class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import List, Optional, Union

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from .argument import Argument, contains_invalid_characters, IoDirection
from .data_handler import DataHandler

FiltersList = Optional[Union[List[str], str]]


class DatasetArgument(Argument):
    """Dataset tool argument."""
    def __init__(
        self,
        data_handler: DataHandler,
        name: str,
        description: str = '',
        io_direction: Optional[IoDirection] = IoDirection.INPUT,
        optional: Optional[bool] = False,
        value: Optional[str] = None,
        hide: Optional[bool] = False,
        filters: FiltersList = None
    ):
        """Construct a dataset argument.

        Args:
            data_handler (DataHandler): Data handler to give list of potential datasets.
            name (Optional[str]): Python friendly argument name.
            description (Optional[str]): User friendly description of the argument.
            io_direction (Optional[IoDirection]): IO Direction of the argument (input or output).
            optional (Optional[bool]): Is the argument optional?
            value (Optional[str]): Default value.
            hide (Optional[bool]): Should the argument be hidden (True) or visible?
            filters (FiltersList): List of filters for allowed dataset types.
        """
        super().__init__(name, description, io_direction, optional, value, hide)
        self.data_handler = data_handler
        if isinstance(filters, str):
            filters = [filters]
        elif filters is None:
            filters = []
        self.filters = set(filters)
        self.choices = None

    def _get_type(self) -> str:
        """Get a string representing the argument type.

        Returns:
            (str): The argument type.
        """
        return 'dataset'

    def _set_value(self, value: str) -> None:
        """Set the argument value.

        Args:
            value (str): The new argument value.
        """
        self._value = None if value is None else str(value)

    def get_interface_info(self) -> Optional[dict[str, object]]:
        """Get interface info for argument to be used in settings dialog.

        Returns:
            Dictionary of interface info.
        """
        interface_info = {
            'type': 'StringSelector' if self.io_direction == IoDirection.INPUT else 'String'
        }
        interface_info = interface_info | super().get_interface_info()
        if self.io_direction == IoDirection.INPUT:
            dataset_list = self.data_handler.get_available_datasets(self.filters).copy()
            dataset_list.insert(0, Argument.NONE_SELECTED)
            if self.value not in dataset_list:
                interface_info['value'] = Argument.NONE_SELECTED
            self.choices = dataset_list
            interface_info['choices'] = dataset_list
        return interface_info

    def to_dict(self) -> dict:
        """Convert an argument to a dictionary.

        Returns:
            (dict): The object values as a dictionary.
        """
        values = {'__class__': 'DatasetArgument'}
        super_values = super(DatasetArgument, self).to_dict()
        values.update(super_values)
        if len(self.filters) > 0:
            filters = list(self.filters)
            filters.sort()
            self._add_key_value('filters', filters, values)
        self._add_key_value('uuid', self.data_handler.get_uuid_from_dataset_name(self.text_value), values)
        return values

    def add_filter(self, filter_type: str) -> None:
        """Update filters to include the given filter type.

        Args:
            filter_type (str): The filter type to be included.
        """
        self.filters.add(filter_type)

    def adjust_value_from_results(self) -> None:
        """Adjust the value obtained when running from previous results."""
        if self.io_direction == IoDirection.INPUT and \
                self.value not in self.data_handler.get_available_datasets(self.filters):
            self.value = None

    def enable_for_grid(self, grid_argument):
        """Change choices to one grids available datasets.

        Args:
            grid_argument (GridArgument): The grid argument.
        """
        grid_name = grid_argument.value
        if self.choices is None:
            self.choices = []
        self.choices.clear()
        self.choices.append(Argument.NONE_SELECTED)
        if grid_name:
            # fill in dataset list for the grid
            datasets = self.data_handler.get_grid_datasets(grid_name)
            for dataset in datasets:
                if dataset.find(f'{grid_name}/') == 0:
                    self.choices.append(dataset)
            # if old value not in list then set it to None
            if self.value not in self.choices:
                self.value = None

    def validate(self):
        """Validate the argument.

        Returns:
            (str): An error string if invalid or None.
        """
        result = super().validate()
        if result is not None:
            return result  # Did not specify when required.

        return contains_invalid_characters(self, allow_forward_slash=True)
