"""CulvertCsvIo class."""
# 1. Standard python modules
import os
import csv

# 2. Third party modules
import numpy as np
import pandas as pd

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as xmf

# 4. Local modules


"""
Columns of the culvert CSV:
    ID, Type, Ignore, UCS, Len_or_ANA, n_or_n_F, US_Invert, DS_Invert, Form_Loss, pBlockage, Inlet_Type, Conn_1d_2d,
    Conn_No, Width_or_Dia, Height_or_WF, Number_of, Height_Cont, Width_Cont, Entry_Loss, Exit_Loss

    The column names are arbitrary. Need to use positional column index
"""
CULVERT_COLUMNS = {  # key=variable_name: value=(dtype, default)
    # Keys is user-friendly text in GUI
    'ID': (np.int32, -1),  # Implicit, doesn't show up in GUI or stored in data
    'Type': (np.int32, 1),  # 1 = circular, 2 = rectangular, 4 = gated circular (unidirectional), 5 = gated rectangular
    'Ignore': (np.int32, 0),
    'UCS': (np.int32, -1),  # Currently unused
    'Len_or_ANA': (np.float64, 0.0),
    'n_or_n_F': (np.float64, 0.0),
    'US_Invert': (np.float64, 0.0),
    'DS_Invert': (np.float64, 0.0),
    'Form_Loss': (np.float64, 0.0),
    'pBlockage': (np.int32, 0),
    'Inlet_Type': (np.int32, -1),  # Currently unused
    'Conn_2D': (np.int32, -1),  # Currently unused
    'Conn_No': (np.int32, -1),  # Currently unused
    'Width_or_Dia': (np.float64, 0.0),
    'Height_or_WF': (np.float64, 0.0),
    'Number_of': (np.int32, 0),
    'Height_Cont': (np.float64, 0.6),
    'Width_Cont': (np.float64, 1.0),
    'Entry_Loss': (np.float64, 0.5),
    'Exit_Loss': (np.float64, 1.0),
}
UNUSED_COLUMNS = {
    'ucs',  # Lowercase so it matches the variable name in the xarray Dataset
    'inlet_type',
    'conn_2d',
    'conn_no',
}
CULVERT_COL_ID = 0
UNSED_COL_VAL = -1


def write_culvert_csv(filename, data):
    """Write all culvert data to a global culvert CSV.

    Args:
        filename (str): Path to the CSV file
        data (StructureData): All structure data, from which culvert data will be extracted
    """
    # Filter out all non-culvert structures
    culverts = data.arcs.where(data.arcs.flux_function == 'Culvert', drop=True)

    # Create the global culvert CSV if needed
    with open(filename, 'w', newline='') as csvfile:
        writer = csv.writer(csvfile)
        # Write Header
        writer.writerow(CULVERT_COLUMNS.keys())
        for i in range(0, len(culverts.comp_id)):
            # Write the culvert's attributes to the CSV
            row = [culverts.comp_id[i].item()]  # Use comp_id as the culvert identifier
            for att_name in CULVERT_COLUMNS.keys():
                if att_name.lower() != "id":  # Skip the ID column, since we are using comp_id instead
                    row.append(culverts[att_name.lower()][i].item())
            writer.writerow(row)


class CulvertCsvReader:
    """Class for reading, updating, and writing culvert CSV files."""
    def __init__(self):
        """Constructor."""
        self._run_folder = ''
        self._files = {}  # {filename: DataFrame}

    def switch_simulation(self, fvc_filename):
        """Switch the current control file being read.

        Args:
            fvc_filename (str): Path to the new simulations fvc filename
        """
        self._run_folder = os.path.dirname(fvc_filename)

    def read_culvert_csv(self, filename, culvert_id, atts):
        """Read a culvert CSV.

        Args:
            filename (str): Path to the CSV file
            culvert_id (int): ID of the culvert in the CSV file
            atts (dict): Attribute dict to update
        """
        # Read the CSV if needed and cache it
        filename = xmf.resolve_relative_path(self._run_folder, filename).lower()
        if not os.path.isfile(filename):
            raise RuntimeError(f'Unable to find referenced culvert CSV file: {filename}')
        df = self._files.get(filename, None)
        if df is None:  # Read this file if we haven't already
            df = pd.read_csv(filename)
            df.columns = [col_name.lower() for col_name in CULVERT_COLUMNS.keys()]
            self._files[filename] = df

        # Slice the DataFrame to the row with the desired culvert id
        df = df.loc[df[df.columns[CULVERT_COL_ID]] == culvert_id]
        if df.empty:  # Should have found the row they asked for
            raise ValueError(f'Unable to find culvert {culvert_id} in culvert CSV {filename}.')
        # Populate the culvert's attributes from the DataFrame
        for att_name in df.columns:
            atts[att_name] = UNSED_COL_VAL if att_name in UNUSED_COLUMNS else df[att_name].iloc[0]

