"""Class to manage arc assignments for BC and structure coverages."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class StructureManager:
    """This class is used to keep track of which geometric arcs have been assigned as structures.

    We need to do this because the `BD` flag in the GIS shapefiles is NULL for all structures. The `BD` flag is also
    NULL if the BC arc has no attributes. In that case, the arc becomes a monitor line that TUFLOWFV calculates flux
    over.
    """
    def __init__(self):
        """Constructor."""
        self._used_structure_arcs = set()
        self._used_structure_polygons = set()
        self._structure_arc_locations = {}  # {struct_id: (data_objects)}
        self._structure_arc_atts = {}  # {struct_id: {variable: value}}
        self._structure_polygon_locations = {}  # {struct_id: (data_objects)}
        self._structure_polygon_atts = {}  # {struct_id: (data_objects)}
        self.arc_struct_id_to_feature_id = {}
        self.arc_struct_id_to_feature_name = {}
        self.poly_struct_id_to_feature_id = {}
        self.poly_struct_id_to_feature_name = {}
        self.next_point_id = 1

    def add_arc_structure(self, name):
        """Assign an arc as a structure.

        Args:
            name (str): The name/id of the arc to add
        """
        self._used_structure_arcs.add(str(name))

    def add_arc_structure_atts(self, name, atts):
        """Assign an arc as a structure.

        Args:
            name (str): The name/ID of the arc
            atts (dict): The attributes of the structure
        """
        self._structure_arc_atts[str(name)] = atts

    def add_arc_structure_location(self, name, arc):
        """Assign an arc as a structure.

        Args:
            name (str): The name/ID of the feature
            arc (Arc): The data_objects Arc
        """
        arc.id = self.num_arc_structure_locations() + 1
        self._structure_arc_locations[name] = arc
        self.arc_struct_id_to_feature_id[name] = arc.id
        self.arc_struct_id_to_feature_name[name] = str(name)

    def is_arc_structure(self, name):
        """Check if an arc has already been assigned to a structure.

        Args:
            name (str): The name/id of the arc to check

        Returns:
            bool: True if the arc has already been used
        """
        return str(name) in self._used_structure_arcs

    def num_arc_structure_locations(self):
        """Get the number of arc locations currently stored.

        Returns:
            int: See description
        """
        return len(self._structure_arc_locations)

    def get_all_arc_locations(self):
        """Get all the locations of the arcs keyed by name.

        Returns:
            dict: See description
        """
        return self._structure_arc_locations

    def get_all_arc_atts(self):
        """Get all the attributes of the arcs keyed by name.

        Returns:
            dict: See description
        """
        return self._structure_arc_atts

    def add_poly_structure(self, name):
        """Assign a polygon as a structure.

        Args:
            name (str): The name/id of the polygon to add
        """
        self._used_structure_polygons.add(str(name))

    def add_poly_structure_atts(self, name, atts):
        """Assign an attributes for a polygon structure.

        Args:
            name (str): The name/ID of the arc
            atts (dict): The attributes of the structure
        """
        self._structure_polygon_atts[str(name)] = atts

    def add_poly_structure_location(self, name, poly):
        """Assign a polygon as a structure.

        Args:
            name (str): The name/ID of the polygon
            poly (Polygon): The data_objects Polygon
        """
        poly.id = self.num_poly_structure_locations() + 1
        self._structure_polygon_locations[name] = poly
        self.poly_struct_id_to_feature_id[name] = poly.id
        self.poly_struct_id_to_feature_name[name] = str(name)

    def is_poly_structure(self, name):
        """Check if a polygon has already been assigned to a structure.

        Args:
            name (str): The name/id of the polygon to check

        Returns:
            bool: True if the polygon has already been used
        """
        return str(name) in self._used_structure_polygons

    def num_poly_structure_locations(self):
        """Get the number of polygon locations currently stored.

        Returns:
            int: See description
        """
        return len(self._structure_polygon_locations)

    def get_all_poly_locations(self):
        """Get all the locations of the polygons keyed by name.

        Returns:
            dict: See description
        """
        return self._structure_polygon_locations

    def get_all_poly_atts(self):
        """Get all the attributes of the polygons keyed by name.

        Returns:
            dict: See description
        """
        return self._structure_polygon_atts

    def clear(self):
        """Clear all the data after a simulation has been read."""
        self._used_structure_arcs = set()
        self._structure_arc_locations = {}
        self._used_structure_polygons = set()
        self._structure_polygon_locations = {}  # {struct_id: (data_objects)}
        self._structure_polygon_atts = {}  # {struct_id: {variable: value}}
        self._structure_arc_atts = {}
        self.poly_struct_id_to_feature_name = {}
        self.arc_struct_id_to_feature_id = {}
        self.arc_struct_id_to_feature_name = {}
        self.poly_struct_id_to_feature_id = {}
