"""Qt table view for defining gridded BCs."""
# 1. Standard python modules

# 2. Third party modules
import numpy as np
import pandas as pd
from PySide2.QtCore import QItemSelectionModel
from PySide2.QtWidgets import QAbstractItemView, QHeaderView

# 3. Aquaveo modules

# 4. Local modules
from xms.tuflowfv.gui import gridded_bcs_filter_model as gbfm
from xms.tuflowfv.gui.add_gridded_bc_button_delegate import AddGriddedBcButtonDelegate
from xms.tuflowfv.gui.define_bc_grid_button_delegate import DefineBcGridButtonDelegate, run_grid_definition_dialog
from xms.tuflowfv.gui.tuflowfv_table import TuflowfvTableWidget


class GriddedBcsTableWidget(TuflowfvTableWidget):
    """The gridded BCs table widget."""

    def __init__(self, sim_data, parent, time_formats):
        """Construct the widget.

        Args:
            sim_data (BcData): The model data
            parent (QDialog): The parent dialog
            time_formats (tuple(str)): The current SMS user preference for formatting absolute datetimes. Should have
                specifiers for (strftime, qt)
        """
        self.sim_data = sim_data
        self.time_formats = time_formats
        dummy_data = np.full(len(self.sim_data.grid_definitions.coords['grid_id']), '')
        data = {'Grid Definition': dummy_data, 'Boundaries': dummy_data}
        self.data_frame = pd.DataFrame(data)
        self.data_frame.index += 1
        defaults = {'Grid Definition': '', 'Boundaries': ''}
        super().__init__(parent=parent, data_frame=self.data_frame, select_col=gbfm.COL_IDX_DEFINE_GRID,
                         default_values=defaults)
        self.setup_ui()

    def setup_ui(self):
        """Add the table widget and initialize the model."""
        define_delegate = DefineBcGridButtonDelegate(parent=self.parent(), sim_data=self.sim_data)
        add_bcs_delegate = AddGriddedBcButtonDelegate(parent=self.parent(), sim_data=self.sim_data,
                                                      time_formats=self.time_formats)
        delegates = {gbfm.COL_IDX_DEFINE_GRID: define_delegate, gbfm.COL_IDX_ADD_BCS: add_bcs_delegate}
        filter_model = gbfm.GriddedBcsFilterModel(self)
        super()._setup_ui(column_delegates=delegates, stretch_last_section=True, fixed_size=False,
                          filter_model=filter_model)
        self.table_view.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        self.table_view.setSelectionMode(QAbstractItemView.SingleSelection)
        self.table_view.setSelectionBehavior(QAbstractItemView.SelectRows)

    def on_btn_add_row(self):
        """Called when a new row is added to the table."""
        super().on_btn_add_row()
        grid_id = self.sim_data.add_grid_definition()
        run_grid_definition_dialog(sim_data=self.sim_data, grid_id=grid_id, parent=self.parent())

    def on_btn_delete_row(self):
        """Called when a row is removed from the table."""
        indices = self.table_view.selectionModel().selectedIndexes()
        next_select_row = -1
        delete_rows = {index.row() if index.isValid() else -1 for index in indices}
        delete_count = 0
        delete_grid_ids = []
        for index in delete_rows:
            if index < 0:
                continue
            delete_row = index - delete_count
            next_select_row = delete_row - 1
            delete_grid_ids.append(self.sim_data.grid_definitions.coords['grid_id'][index].item())
            self.model.removeRows(delete_row, 1)
            delete_count += 1
        self.sim_data.grid_definitions = self.sim_data.grid_definitions.where(
            ~self.sim_data.grid_definitions.grid_id.isin(delete_grid_ids), drop=True
        )
        self.sim_data.gridded_bcs.bcs = self.sim_data.gridded_bcs.bcs.where(
            ~self.sim_data.gridded_bcs.bcs.grid_id.isin(delete_grid_ids), drop=True
        )
        if next_select_row >= 0:
            select_index = self.table_view.model().index(next_select_row, self.select_col)
            self.table_view.selectionModel().setCurrentIndex(
                select_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
            )
