"""Base class for Qt table views used in xmstuflowfv."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.tuflowfv.gui.output_coverages_filter_model import OutputCoveragesFilterModel
from xms.tuflowfv.gui.tree_item_selector_delegate import TreeItemSelectorDelegate
from xms.tuflowfv.gui.tuflowfv_table import TuflowfvTableWidget


COVERAGE_COL_IDX = 0
COVERAGE_ROW_IDX = 1


class OutputCoveragesTable(TuflowfvTableWidget):
    """Table widget base class."""
    def __init__(self, parent, data_frame, data, tree_node, blocks_table):
        """Construct the widget.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            data_frame (pandas.DataFrame): The model data.
            data (SimData): The simulation dataset
            blocks_table (TuflowfvTableWidget): The output blocks table widget
            tree_node (TreeNode): Root of the project explorer tree
        """
        super().__init__(parent=parent, data_frame=data_frame, select_col=COVERAGE_COL_IDX, default_values={'uuid': ''})
        self._data = data
        self._tree_node = tree_node
        self._blocks_table = blocks_table
        self._setup_ui_derived()
        self.remove_uuids = []
        self.remove_indices = []

    def _setup_ui_derived(self):
        """Setup the table widget."""
        delegate = TreeItemSelectorDelegate(tree_type='TI_COVER', tree_node=self._tree_node, parent=self.parent())
        filter_model = OutputCoveragesFilterModel(parent=self.parent())
        super()._setup_ui(
            column_delegates={COVERAGE_COL_IDX: delegate},
            stretch_last_section=True,
            fixed_size=False,
            filter_model=filter_model
        )
        self.table_view.horizontalHeader().setVisible(False)
        self.table_view.setColumnHidden(COVERAGE_ROW_IDX, True)

    def on_btn_add_row(self):
        """Called when a new row is added to the table."""
        super().on_btn_add_row()
        qt_cov_row_index = self.model.rowCount() - 1
        indices = self._blocks_table.table_view.selectionModel().selectedRows()
        if len(indices) != 1 or not indices[0].isValid():
            return
        qt_block_row_index = indices[0].row()
        row_index = self._blocks_table.model.data_frame['row_index'].iat[qt_block_row_index]
        self.model.data_frame['row_index'].iat[qt_cov_row_index] = row_index

    def on_btn_delete_row(self):
        """Override to store deleted coverages from the table."""
        indices = self.table_view.selectionModel().selectedIndexes()
        row_indices = {index.row() if index.isValid() else -1 for index in indices}
        for row in row_indices:
            if row < 0:
                continue
            self.remove_uuids.append(self.model.data_frame['uuid'].iloc[row])
            self.remove_indices.append(self.model.data_frame['row_index'].iloc[row])
        super().on_btn_delete_row()
